package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonArray;
import com.google.common.primitives.UnsignedLong;

/**
 * @author amaslak
 */

public class DbShardsIdsLogParser implements LogParser {

    public static final String DATE_PATTERN_PERL = "yyyy-MM-dd'T'HH:mm:ssX";
    public static final String DATE_PATTERN_JAVA = "yyyy-MM-dd:HH:mm:ss";
    public final SimpleDateFormat dateFormatPerl = new SimpleDateFormat(DATE_PATTERN_PERL);
    public final SimpleDateFormat dateFormatJava = new SimpleDateFormat(DATE_PATTERN_JAVA);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(reqid)";

    public static List<String> additionalPrimaryKeys = Arrays.asList(semplName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("source", ColumnType.String),
        new Column("key", ColumnType.String),
        new Column("reqid", ColumnType.Int64),
        new Column("host", ColumnType.String),
        new Column("insert_data", ColumnType.String),
        new Column("ids", ColumnType.ArrayUInt64)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private String extractJsonElement(JsonObject json, String key) {
        return (json != null && json.get(key) != null) ? json.get(key).getAsString() : "";
    }

    private static final Gson gson = new Gson();

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    public static boolean isJSONValid(String jsonInString) {
        try {
            gson.fromJson(jsonInString, Object.class);
            return true;
        } catch(com.google.gson.JsonSyntaxException ex) {
            return false;
        }
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {

        JsonObject element;
        String source;
        String host;
        Date date;

        if (isJSONValid(line)) {
            element = gson.fromJson(line, JsonObject.class);
            date = dateFormatJava.parse(element.get("log_time").getAsString());
            source = "PPC.dbshards_ids.log";
            host = element.get("host").getAsString();
        } else {
            String[] matcher = line.split(" ", 10);
            element = gson.fromJson(matcher[matcher.length - 1], JsonObject.class);
            date = dateFormatPerl.parse(matcher[1]);
            source = matcher[3];
            host = matcher[2];
        }

        long reqid = element.get("reqid").getAsLong();
        String key = element.get("key").toString().replace("\"", "");
        String insert_data = element.get("insert_data").toString();
        JsonArray jArray = element.get("ids").getAsJsonArray();
        List<UnsignedLong> keys = new ArrayList<>();

        jArray.forEach(jsonElement -> {
            keys.add(UnsignedLong.valueOf(jsonElement.getAsString()));
        });

        context.write(date, source, key, reqid, host, insert_data, keys);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

}

