package ru.yandex.market.logshatter.parser.direct;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static java.util.Collections.singletonList;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsBooleanOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsLongOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;


public class EssLogicObjectsParser implements LogParser {

    public static final Column DATE_COLUMN = new Column("binlog_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("binlog_time", ColumnType.DateTime);

    private static String sampleBy = "intHash64(reqid)";
    private static List<String> sampleColumns = singletonList(sampleBy);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, sampleBy, 8192);

    private static final Gson gson = new Gson();

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add("topic");
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(sampleColumns);
        return aggrKeys;
    }

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("reqid", ColumnType.Int64),
        new Column("gtid", ColumnType.String),
        new Column("seqNo", ColumnType.Int64),
        new Column("source", ColumnType.String),
        new Column("topic", ColumnType.String),
        new Column("logic_object.attr", ColumnType.ArrayString, null, null, "ZSTD(5)"),
        new Column("logic_object.value", ColumnType.ArrayString, null, null, "ZSTD(5)")
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 1);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject logicObjectWithSystemInfo = gson.fromJson(line, JsonObject.class);

        boolean isPingObject = getAsBooleanOrDefault(logicObjectWithSystemInfo, "isPingObject");
        if (isPingObject) {
            return;
        }
        Date date = Date.from(Instant.ofEpochSecond(logicObjectWithSystemInfo.get("utcTimestamp").getAsLong()));

        Long reqId = getAsLongOrDefault(logicObjectWithSystemInfo, "reqId");
        String gtid = getAsStringOrDefault(logicObjectWithSystemInfo, "gtid");
        Long seqNo = getAsLongOrDefault(logicObjectWithSystemInfo, "seqNo");
        String source = getAsStringOrDefault(logicObjectWithSystemInfo, "source");
        String topic = context.getLogBrokerTopic();

        for (JsonElement logicObject : logicObjectWithSystemInfo.get("logicObjectsList").getAsJsonArray()) {
            List<String> columns = new ArrayList<>();
            List<String> values = new ArrayList<>();
            logicObject.getAsJsonObject().entrySet().forEach(
                entry -> {
                    columns.add(entry.getKey());
                    if (entry.getValue().isJsonPrimitive() && ((JsonPrimitive) entry.getValue()).isString()) {
                        values.add(entry.getValue().getAsString());
                    } else {
                        values.add(entry.getValue().toString());
                    }
                }
            );

            Object[] result = new Object[]{
                reqId,
                gtid,
                seqNo,
                source,
                topic,
                columns,
                values
            };

            context.write(date, result);
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
