package ru.yandex.market.logshatter.parser.direct;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.google.gson.Gson;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.direct.logformat.EssModerationFormat;

import static java.util.Arrays.asList;


public class EssModerationLogParser implements LogParser {
    private static final String DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateTimeFormat = new SimpleDateFormat(DATE_TIME_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String SAMPLE_COLUMN_NAME = "intHash64(span_id)";

    private static final List<String> PRIMARY_KEYS = asList(DATE_COLUMN.getName(), TIMESTAMP_COLUMN.getName(),
        SAMPLE_COLUMN_NAME);
    private static final String PART_NAME = DATE_COLUMN.getName();

    private static final MergeTree
        DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + PART_NAME + ")", PRIMARY_KEYS, SAMPLE_COLUMN_NAME, 8192);

    private static final Gson gson = new Gson();

    private static List<Column> columnsStruct = new ArrayList<>(asList(
        new Column("cid", ColumnType.Int64),
        new Column("pid", ColumnType.Int64),
        new Column("bid", ColumnType.Int64),
        new Column("span_id", ColumnType.UInt64),
        new Column("action", ColumnType.String),
        new Column("source", ColumnType.String),
        new Column("success", ColumnType.String),
        new Column("data", ColumnType.String)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 3);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        EssModerationFormat essModerationFormat = gson.fromJson(line, EssModerationFormat.class);

        context.write(dateTimeFormat.parse(essModerationFormat.getLogTime()),
            essModerationFormat.getCid(),
            essModerationFormat.getPid(),
            essModerationFormat.getBid(),
            essModerationFormat.getSpanId(),
            Optional.ofNullable(essModerationFormat.getAction()).orElse(""),
            Optional.ofNullable(essModerationFormat.getSource()).orElse(""),
            essModerationFormat.isSuccess(),
            essModerationFormat.getData()
        );
    }
}
