package ru.yandex.market.logshatter.parser.direct;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static java.util.Arrays.asList;

public class JsTemplaterLogParser implements LogParser {
    private static final Gson gson = new Gson();
    private static final List<String> REMOVE_PROPERTIES = asList(
            "scarab:type", "timestamp",
            "scarab:version", "scarab:stable", "scarab:format"
    );

    @Override
    public TableDescription getTableDescription() {
        Column dateColumn = new Column("log_date", ColumnType.Date);
        Column timestampColumn = new Column("log_time", ColumnType.DateTime);

        return new TableDescription(
                new ArrayList<>(asList(
                        dateColumn,
                        timestampColumn,
                        new Column("log_time_millis", ColumnType.UInt16),
                        new Column("log_type", ColumnType.String),
                        new Column("host", ColumnType.String),
                        new Column("reqid", ColumnType.UInt64),
                        new Column("data", ColumnType.String, null,null, "ZSTD(5)")
                )),
                new MergeTree(
                        String.format("toYYYYMM(%s)", dateColumn.getName()),
                        asList(dateColumn.getName(), timestampColumn.getName()),
                        null,
                        8192));
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject rec = gson.fromJson(line, JsonObject.class);

        String logType = rec.get("scarab:type").getAsString();

        long timestamp = rec.get("timestamp").getAsLong();
        Date date = new Date(timestamp);

        int logTimeMillis = (int) (timestamp % 1000);

        long reqid;
        if ("COMMON_FAILURE_EVENT".equals(logType)) {
            reqid = Long.parseLong(
                    rec.getAsJsonObject("failed_event_info")
                            .getAsJsonObject("properties")
                            .get("requestId").getAsString()
            );
        } else {
            reqid = Long.parseLong(rec.get("requestId").getAsString());
        }

        REMOVE_PROPERTIES.forEach(rec::remove);

        context.write(
                date,
                logTimeMillis,
                logType,
                context.getHost(),
                reqid,
                rec.toString()
        );
    }
}
