package ru.yandex.market.logshatter.parser.direct;


import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import com.google.common.collect.ObjectArrays;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsLongOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;

public class MailsLogParser implements LogParser {

    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(reqid)";

    public static List<String> additionalPrimaryKeys = Collections.singletonList(semplName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 1024);

    private static final Gson gson = new Gson();

    private static List<Column> columnsStruct = new ArrayList<>(Arrays.asList(
        new Column("host", ColumnType.String),
        new Column("service", ColumnType.String),
        new Column("method", ColumnType.String),
        new Column("reqid", ColumnType.Int64),

        new Column("email", ColumnType.String),
        new Column("template_name", ColumnType.String),
        new Column("subject", ColumnType.String),
        new Column("content", ColumnType.String),
        new Column("client_id", ColumnType.Int64)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    private Object[] getMailInfoAsArray(JsonObject json) {
        return new Object[]{
            getAsStringOrDefault(json, "email"),
            getAsStringOrDefault(json, "template_name"),
            getAsStringOrDefault(json, "subject"),
            getAsStringOrDefault(json, "content"),
            getAsLongOrDefault(json, "client_id")
        };
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject log = gson.fromJson(line, JsonObject.class);
        String logTime = log.get("log_time").getAsString();
        Date date = dateFormat.parse(logTime);

        Object[] mainLogInfo = new Object[]{
            getAsStringOrDefault(log, "log_hostname"),
            getAsStringOrDefault(log, "service"),
            getAsStringOrDefault(log, "method"),
            getAsLongOrDefault(log, "reqid"),
        };

        JsonElement data = log.get("data");
        if (data.isJsonArray()) {
            for (JsonElement element : data.getAsJsonArray()) {
                Object[] dataArray = getMailInfoAsArray(element.getAsJsonObject());
                context.write(date, ObjectArrays.concat(mainLogInfo, dataArray, Object.class));
            }
        } else {
            Object[] dataArray = getMailInfoAsArray(data.getAsJsonObject());
            context.write(date, ObjectArrays.concat(mainLogInfo, dataArray, Object.class));
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
