package ru.yandex.market.logshatter.parser.direct;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ObjectArrays;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsDoubleOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsLongOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;

public class MetricsLogParser implements LogParser {
    private static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    private static final Gson gson = new Gson();

    @Override
    public TableDescription getTableDescription() {
        Column dateColumn = new Column("log_date", ColumnType.Date);
        Column timestampColumn = new Column("log_time", ColumnType.DateTime);
        Column metricNameColumn = new Column("metric_name", ColumnType.String);
        String sampleName = "intHash64(reqid)";

        return new TableDescription(
            new ArrayList<>(Arrays.asList(
                dateColumn,
                timestampColumn,
                new Column("service", ColumnType.String),
                new Column("method", ColumnType.String),
                new Column("host", ColumnType.String),
                new Column("reqid", ColumnType.UInt64),
                new Column("ip", ColumnType.String),

                metricNameColumn,
                new Column("metric_value", ColumnType.Float64),
                new Column("context.key", ColumnType.ArrayString),
                new Column("context.value", ColumnType.ArrayString)
            )),
            new MergeTree(
                String.format("toYYYYMM(%s)", dateColumn.getName()),
                Arrays.asList(dateColumn.getName(), metricNameColumn.getName(), timestampColumn.getName(), sampleName),
                sampleName,
                8192));
    }

    private Object[] getMetricInfoAsArray(JsonObject metric) {
        Set<Map.Entry<String, JsonElement>> rowsEntrySet = metric.getAsJsonObject("context").entrySet();
        Map<String, JsonElement> rowKV = new HashMap<>();
        for (Map.Entry<String, JsonElement> kv : rowsEntrySet) {
            rowKV.put(kv.getKey(), kv.getValue());
        }
        List<String> contextKeys = new ArrayList<>(rowKV.keySet());
        List<String> contextValues = new ArrayList<>(contextKeys.size());
        contextKeys.sort(String.CASE_INSENSITIVE_ORDER);
        for (String k : contextKeys) {
            JsonElement valueObject = rowKV.get(k);
            String v;
            if (valueObject.isJsonPrimitive() && !valueObject.isJsonNull()) {
                v = valueObject.getAsString();
            } else {
                v = valueObject.toString();
            }
            contextValues.add(v);
        }

        return new Object[]{
            getAsStringOrDefault(metric, "name"),
            getAsDoubleOrDefault(metric, "value"),
            contextKeys,
            contextValues
        };
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject record = gson.fromJson(line, JsonObject.class);

        String logType = record.get("log_type").getAsString();
        if (!logType.equals("metrics")) {
            return;
        }

        Date date = dateFormat.parse(record.get("log_time").getAsString());
        Object[] mainLogInfo = new Object[]{
            getAsStringOrDefault(record, "service"),
            getAsStringOrDefault(record, "method"),
            getAsStringOrDefault(record, "log_hostname"),
            getAsLongOrDefault(record, "reqid"),
            getAsStringOrDefault(record, "ip")
        };

        JsonElement data = record.get("data");
        if (data.isJsonArray()) {
            for (JsonElement metric : data.getAsJsonArray()) {
                Object[] dataArray = getMetricInfoAsArray(metric.getAsJsonObject());
                context.write(date, ObjectArrays.concat(mainLogInfo, dataArray, Object.class));
            }
        } else {
            Object[] dataArray = getMetricInfoAsArray(data.getAsJsonObject());
            context.write(date, ObjectArrays.concat(mainLogInfo, dataArray, Object.class));
        }
    }
}
