package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;


import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;


public class ModerateLogParser implements LogParser {
    private static final int LINE_PATHS_NUMBER = 4;

    private static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    private static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    private static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String SAMPLE_COLUMN_NAME = "intHash64(cid)";

    private static final List<String> ADDITIONAL_PRIMARY_KEYS = Arrays.asList(SAMPLE_COLUMN_NAME);
    private static final List<String> PRIMARY_KEYS = getPrimary();
    private static final String PART_COLUMN_NAME = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE = new MergeTree(
        String.format("toYYYYMM(%s)", PART_COLUMN_NAME),
        PRIMARY_KEYS,
        SAMPLE_COLUMN_NAME,
        8192
    );

    private static final List<Column> COLUMNS_STRUCT = new ArrayList<Column>(Arrays.asList(
        new Column("cid", ColumnType.Int64),
        new Column("bid", ColumnType.Int64),
        new Column("pid", ColumnType.Int64),
        new Column("data_type", ColumnType.String),
        new Column("uuid", ColumnType.String),
        new Column("metadata", ColumnType.String),
        new Column("source", ColumnType.String),
        new Column("data", ColumnType.String, null, null, "ZSTD(5)")
    ));

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, COLUMNS_STRUCT);

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(ADDITIONAL_PRIMARY_KEYS);
        return aggrKeys;
    }

    private static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static long safeIdParse(String id) {
        try {
            long value = Long.parseLong(id);
            if (value < 0) {
                value = 0;
            }
            return value;
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    private static String getStringParam(String paramName, Map<String, String> params) {
        String value = params.getOrDefault(paramName, "");

        if (value.isEmpty()) {
            throw new IllegalStateException(String.format("Empty value for parameter %s", paramName));
        } else {
            return value;
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        String[] paths = line.split("\t");
        if (paths.length < LINE_PATHS_NUMBER) {
            throw new IllegalArgumentException(String.format("Can't parse line: %s", line));
        }

        Date date = dateFormat.parse(String.format("%s %s", paths[0], paths[1]));
        String metadata = paths[2];
        String data = paths[3];

        Map<String, String> metadataParams = new HashMap<>();

        for (String param : metadata.split(",")) {
            int splitIndex = param.indexOf('=');
            if (splitIndex > 0) {
                metadataParams.put(param.substring(0, splitIndex), param.substring(splitIndex + 1));
            }
        }

        long cid = safeIdParse(metadataParams.getOrDefault("cid", "0"));
        long bid = safeIdParse(metadataParams.getOrDefault("bid", "0"));
        long pid = safeIdParse(metadataParams.getOrDefault("pid", "0"));
        String dataType = getStringParam("data_type", metadataParams);
        String uuid = getStringParam("uuid", metadataParams);

        context.write(
            date,
            cid,
            bid,
            pid,
            dataType,
            uuid,
            metadata,
            "file:" + context.getHost() + context.getFile().toString(),
            data
        );
    }
}
