package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import java.text.SimpleDateFormat;
import java.util.TimeZone;

import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class MysqlAuditLogParser implements LogParser {
    public static final String DATE_PATTERN = "yyyy-MM-dd'T'HH:mm:ss z";

    public final SimpleDateFormat dateFormat;
    {
        dateFormat = new SimpleDateFormat(DATE_PATTERN);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "sipHash64(ip)";

    public static List<String> primaryKeys = Arrays.asList(DATE_COLUMN.getName(), semplName);

    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static final Gson gson = new Gson();

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("name", ColumnType.String),
        new Column("record", ColumnType.String),
        new Column("hostname", ColumnType.String),
        new Column("connection_id", ColumnType.Int64),
        new Column("status", ColumnType.Int64),
        new Column("user", ColumnType.String),
        new Column("priv_user", ColumnType.String),
        new Column("os_login", ColumnType.String),
        new Column("proxy_user", ColumnType.String),
        new Column("host", ColumnType.String),
        new Column("ip", ColumnType.String),
        new Column("db", ColumnType.String)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        if (!line.startsWith("{")) {
            return; //исключаем любую не json строку audit лога.
        }		
        JsonObject auditRecord = gson.fromJson(line, JsonObject.class);
        JsonObject element = auditRecord.get("audit_record").getAsJsonObject();
        Date date = dateFormat.parse(element.get("timestamp").getAsString());
        String name = element.get("name").getAsString();
        String record = element.get("record").getAsString();
        String hostname = context.getHost();
        long connectionId = element.get("connection_id").getAsLong();
        long status = element.get("status").getAsLong();
        String user = element.get("user").getAsString();
        String privUser = element.get("priv_user").getAsString();
        String osLogin = element.get("os_login").getAsString();
        String proxyUser = element.get("proxy_user").getAsString();
        String host = element.get("host").getAsString();
        String ip = element.get("ip").getAsString();
        String db = element.get("db").getAsString();

        context.write(date, name, record, hostname, connectionId, status, user,
            privUser, osLogin, proxyUser, host, ip, db);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

}
