package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.logshatter.parser.LogParser;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.List;
import java.util.ArrayList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MysqlPtkillLogParser implements LogParser {
    public static final String DATE_PATTERN = "yyyy-MM-dd'T'HH:mm:ss";

    SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(query_id)";

    public static List<String> primaryKeys = Arrays.asList(DATE_COLUMN.getName(), semplName);

    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("hostname", ColumnType.String),
        new Column("query_id", ColumnType.Int64),
        new Column("request_time", ColumnType.Int64),
        new Column("query", ColumnType.String),
        new Column("meta", ColumnType.String),
        new Column("instance", ColumnType.String)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private static String PARSE_REGEX = "(?<query>.*/\\*\\s+(?<meta>.*)\\s+\\*/.*)$";
    private static final Pattern PATTERN = Pattern.compile(PARSE_REGEX);

    private static final Gson gson = new Gson();

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        if (line.isEmpty()) {
            return;
        }

        if (line.startsWith("{")) {
            JsonObject rec = gson.fromJson(line, JsonObject.class);
            String date = rec.get("timestamp").getAsString();
            String hostname = context.getHost();
            String queryType = rec.get("query_type").getAsString();
            long queryId = rec.get("query_id").getAsLong();
            String queryCommand = rec.get("query_command").getAsString();
            long queryTime = rec.get("query_time_second").getAsLong();
            String queryInfo = rec.get("query_info").getAsString();
            String shardName = rec.get("shard_name").getAsString();

            Matcher matcher = PATTERN.matcher(line);
            String meta = (matcher.find()) ? matcher.group("meta") : "";
	                    
            context.write(
		dateFormat.parse(date),
                hostname,
                queryId,
                queryTime,
                queryInfo,
                meta,
                shardName);
        } else {
            throw new IllegalArgumentException("Can't parse line: " + line);
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
