package ru.yandex.market.logshatter.parser.direct;


import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import com.google.common.collect.ObjectArrays;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsFloatOrDefault;
import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;


public class PpcPriceLogParser implements LogParser {

    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(id)";

    public static List<String> additionalPrimaryKeys = Arrays.asList(semplName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static final Gson gson = new Gson();

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("service", ColumnType.String),
        new Column("method", ColumnType.String),
        new Column("reqid", ColumnType.Int64),
        new Column("uid", ColumnType.Int64),
        new Column("host", ColumnType.String),
        new Column("ip", ColumnType.String),
        new Column("cid", ColumnType.Int64),
        new Column("pid", ColumnType.Int64),
        new Column("bid", ColumnType.Int64),
        new Column("id", ColumnType.Int64),
        new Column("type", ColumnType.String),
        new Column("currency", ColumnType.String),
        new Column("price", ColumnType.Float32),
        new Column("price_ctx", ColumnType.Float32)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    private Object[] getPpcPriceLogInfoAsArray(JsonObject json) {
        return new Object[]{
            json.get("cid").getAsLong(),
            json.get("pid").getAsLong(),
            json.get("bid").getAsLong(),
            json.get("id").getAsLong(),
            getAsStringOrDefault(json, "type"),
            getAsStringOrDefault(json, "currency"),
            getAsFloatOrDefault(json, "price"),
            getAsFloatOrDefault(json, "price_ctx")
        };
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject log = gson.fromJson(line, JsonObject.class);
        String logTime = log.get("log_time").getAsString();
        Date date = dateFormat.parse(logTime);

        Object[] mainLogInfo = new Object[]{
            log.get("service").getAsString(),
            getAsStringOrDefault(log, "method"),
            log.get("reqid").getAsLong(),
            log.get("uid").getAsInt(),
            log.get("log_hostname").getAsString(),
            getAsStringOrDefault(log, "ip")
        };

        for (JsonElement ppcPriceLogInfo : log.getAsJsonArray("data")) {
            Object[] ppcPriceLogArray = getPpcPriceLogInfoAsArray(ppcPriceLogInfo.getAsJsonObject());
            context.write(date, ObjectArrays.concat(mainLogInfo, ppcPriceLogArray, Object.class));
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
