package ru.yandex.market.logshatter.parser.direct;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.TimeZone;
import java.util.stream.Stream;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;


public class TraceLogParser implements LogParser {
    public static final ZoneId MSK = ZoneId.of("Europe/Moscow");
    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";

    public final SimpleDateFormat dateFormat;
    {
        dateFormat = new SimpleDateFormat(DATE_PATTERN);
        dateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
    }

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(span_id)";

    public static List<String> primaryKeys =
        Arrays.asList(DATE_COLUMN.getName(), "service", "method", TIMESTAMP_COLUMN.getName(), semplName);
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE =
        new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static final Gson gson = new Gson();

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("log_time_nanos", ColumnType.Int32),
        new Column("format_id", ColumnType.Int64),
        new Column("host", ColumnType.String),
        new Column("service", ColumnType.String),
        new Column("method", ColumnType.String),
        new Column("tags", ColumnType.String),
        new Column("trace_id", ColumnType.Int64),
        new Column("parent_id", ColumnType.Int64),
        new Column("span_id", ColumnType.Int64),
        new Column("chunk_index", ColumnType.Int64),
        new Column("chunk_last", ColumnType.UInt8),
        new Column("span_time", ColumnType.Float32),
        new Column("samplerate", ColumnType.Int64),

        new Column("ela", ColumnType.Float32),
        new Column("cpu_user", ColumnType.Float32),
        new Column("cpu_system", ColumnType.Float32),
        new Column("mem", ColumnType.Float32),

        new Column("profile.func", ColumnType.ArrayString),
        new Column("profile.tags", ColumnType.ArrayString),
        new Column("profile.all_ela", ColumnType.ArrayFloat32),
        new Column("profile.childs_ela", ColumnType.ArrayFloat32),
        new Column("profile.calls", ColumnType.ArrayInt32),
        new Column("profile.obj_num", ColumnType.ArrayInt32),

        new Column("services.service", ColumnType.ArrayString),
        new Column("services.method", ColumnType.ArrayString),
        new Column("services.span_id", ColumnType.ArrayInt64),
        new Column("services.rel_client_send", ColumnType.ArrayFloat32),
        new Column("services.ela", ColumnType.ArrayFloat32),

        new Column("marks.relative_time", ColumnType.ArrayFloat32),
        new Column("marks.message", ColumnType.ArrayString),

        new Column("annotations.key", ColumnType.ArrayString),
        new Column("annotations.value", ColumnType.ArrayString)

    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private Object[] parseTimes(JsonObject times) {
        Float ela = times.get("ela").getAsFloat();
        Float cpuUser = times.get("cu").getAsFloat();
        Float cpuSystem = times.get("cs").getAsFloat();
        Float mem = times.get("mem").getAsFloat();
        return new Object[]{
            ela,
            cpuUser,
            cpuSystem,
            mem
        };
    }

    private List<?>[] parseProfile(JsonArray profile)
    {
        List<String> funcs = new ArrayList<>();
        List<String> tags = new ArrayList<>();
        List<Float> allEla = new ArrayList<>();
        List<Float> childsEla = new ArrayList<>();
        List<Integer> calls = new ArrayList<>();
        List<Integer> objNums = new ArrayList<>();

        for (JsonElement profileLine : profile) {
            JsonArray simpleElements = profileLine.getAsJsonArray();
            int elementIdx = 0;
            funcs.add(simpleElements.get(elementIdx++).getAsString());
            tags.add(simpleElements.get(elementIdx++).getAsString());
            allEla.add(simpleElements.get(elementIdx++).getAsFloat());
            childsEla.add(simpleElements.get(elementIdx++).getAsFloat());
            calls.add(simpleElements.get(elementIdx++).getAsInt());
            objNums.add(simpleElements.get(elementIdx++).getAsInt());
        }
        return new List[]{
            funcs,
            tags,
            allEla,
            childsEla,
            calls,
            objNums
        };
    }

    private List<?>[] parseServices(JsonArray servicesArray)
    {
        List<String> services = new ArrayList<>();
        List<String> methods = new ArrayList<>();
        List<Long> spanIds = new ArrayList<>();
        List<Float> relClientSends = new ArrayList<>();
        List<Float> ela = new ArrayList<>();
        for (JsonElement serviceLine : servicesArray) {
            JsonArray simpleElements = serviceLine.getAsJsonArray();
            int elementIdx = 0;
            services.add(simpleElements.get(elementIdx++).getAsString());
            methods.add(simpleElements.get(elementIdx++).getAsString());
            spanIds.add(simpleElements.get(elementIdx++).getAsLong());
            relClientSends.add(simpleElements.get(elementIdx++).getAsFloat());
            ela.add(simpleElements.get(elementIdx++).getAsFloat());
        }
        return new List[]{
            services,
            methods,
            spanIds,
            relClientSends,
            ela
        };
    }

    private List<?>[] parseMarks(JsonArray marks) {

        List<Float> relativeTimes = new ArrayList<>();
        List<String> messages = new ArrayList<>();
        for (JsonElement markLine : marks) {
            JsonArray simpleElements = markLine.getAsJsonArray();
            int elementIdx = 0;
            relativeTimes.add(simpleElements.get(elementIdx++).getAsFloat());
            messages.add(simpleElements.get(elementIdx).getAsString());
        }
        return new List[]{
            relativeTimes,
            messages
        };
    }

    private List<?>[] parseAnnotations(JsonArray annotations) {
        List<String> keys = new ArrayList<>();
        List<String> values = new ArrayList<>();
        for (JsonElement annotationLine : annotations) {
            JsonArray simpleElements = annotationLine.getAsJsonArray();
            int elementIdx = 0;
            keys.add(simpleElements.get(elementIdx++).getAsString());
            values.add(simpleElements.get(elementIdx++).getAsString());
        }

        return new List[]{
            keys,
            values
        };
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        int valueIdx = 0;
        JsonArray lineJsonArray = gson.fromJson(line, JsonArray.class);

        int formatId = lineJsonArray.get(valueIdx++).getAsInt(); // 3
        if (formatId != 3) {
            throw new IllegalArgumentException("Unknown format_id: " + formatId);
        }
        String dateTime = lineJsonArray.get(valueIdx++).getAsString();

        int pos = dateTime.indexOf('.');
        Instant instant;
        int log_time_nanos = 0;
        if (pos < 0) {
            instant = dateFormat.parse(dateTime).toInstant();
        } else {
            instant = dateFormat.parse(dateTime.substring(0, pos)).toInstant();
            log_time_nanos = DirectParserUtils.parseNanos(dateTime.substring(pos + 1));
        }

        String host = lineJsonArray.get(valueIdx++).getAsString();
        valueIdx++; // pid
        String service = lineJsonArray.get(valueIdx++).getAsString();
        String method = lineJsonArray.get(valueIdx++).getAsString();

        JsonElement tagElement = lineJsonArray.get(valueIdx++);
        String tags = tagElement.isJsonNull() ? "" : tagElement.getAsString();
        long traceId = lineJsonArray.get(valueIdx++).getAsLong();
        long parentId = lineJsonArray.get(valueIdx++).getAsLong();
        long spanId = lineJsonArray.get(valueIdx++).getAsLong();
        int chunkIndex = lineJsonArray.get(valueIdx++).getAsInt();
        int chunkLast = lineJsonArray.get(valueIdx++).getAsBoolean() ? 1 : 0;
        float spanTime = lineJsonArray.get(valueIdx++).getAsFloat();
        int samplerate = lineJsonArray.get(valueIdx++).getAsInt();

        Object[] mainInfo = new Object[]{
            log_time_nanos,
            formatId,
            host,
            service,
            method,
            tags,
            traceId,
            parentId,
            spanId,
            chunkIndex,
            chunkLast,
            spanTime,
            samplerate
        };

        JsonObject jsonObject = lineJsonArray.get(valueIdx++).getAsJsonObject();

        Object[] times = parseTimes(jsonObject.get("times").getAsJsonObject());
        List<?>[] profile = parseProfile(jsonObject.get("profile").getAsJsonArray());
        List<?>[] services = parseServices(jsonObject.get("services").getAsJsonArray());
        List<?>[] marks = parseMarks(jsonObject.get("marks").getAsJsonArray());
        List<?>[] annotations = jsonObject.has("annotations") ?
            parseAnnotations(jsonObject.get("annotations").getAsJsonArray()) :
            Collections.nCopies(2, Collections.emptyList()).toArray(new List<?>[0]);

        Object[] result =
            Stream.of(mainInfo, times, profile, services, marks, annotations)
                .flatMap(Stream::of)
                .toArray(Object[]::new);

        context.write(instant.atZone(MSK), result);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
