package ru.yandex.market.logshatter.parser.direct;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;

public class UaasDataLogParser implements LogParser {

    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateFormatJava = new SimpleDateFormat(DATE_PATTERN);

    private static final Gson gson = new Gson();

    @Override
    public TableDescription getTableDescription() {
        Column dateColumn = new Column("log_date", ColumnType.Date);
        Column timestampColumn = new Column("log_time", ColumnType.DateTime);
        String sampleName = "intHash64(reqid)";

        return new TableDescription(
            new ArrayList<>(Arrays.asList(
                dateColumn,
                timestampColumn,
                new Column("reqid", ColumnType.UInt64),
                new Column("client_id", ColumnType.UInt64),
                new Column("yandexuid", ColumnType.String),
                new Column("exp_boxes", ColumnType.String),
                new Column("features", ColumnType.String)
            )),
            new MergeTree(
                String.format("toYYYYMM(%s)", dateColumn.getName()),
                Arrays.asList(dateColumn.getName(), timestampColumn.getName(), sampleName),
                sampleName,
                8192));
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject record = gson.fromJson(line, JsonObject.class);
        Date date = dateFormatJava.parse(record.get("log_time").getAsString());

        JsonObject data = record.get("data").getAsJsonObject();
        long reqId = record.get("reqid").getAsLong();
        Long clientId = data.get("ClientID").getAsLong();
        String yandexuid = getAsStringOrDefault(data, "yandexuid");
        String expBoxes = data.get("exp_boxes").getAsString();
        String features = data.get("features").getAsString();

        context.write(date, reqId, clientId, yandexuid, expBoxes, features);
    }
}