package ru.yandex.market.logshatter.parser.direct.bsexport;

import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;

import com.google.gson.*;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import static ru.yandex.market.logshatter.parser.direct.DirectParserUtils.getAsStringOrDefault;

public abstract class BaseBsExportLogParser implements LogParser {
    public static final String DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateTimeFormat = new SimpleDateFormat(DATE_TIME_PATTERN);

    @Override
    public TableDescription getTableDescription() {
        Column dateColumn = new Column("log_date", ColumnType.Date);
        Column timestampColumn = new Column("log_time", ColumnType.DateTime);
        Column cidColumn = new Column("cid", ColumnType.Int64);
        String sampleName = "intHash64(cid)";

        return new TableDescription(
                new ArrayList<>(Arrays.asList(
                        dateColumn,
                        timestampColumn,
                        cidColumn,
                        new Column("par_id", ColumnType.Int64),
                        new Column("par_type", ColumnType.String),
                        new Column("data_type", ColumnType.String),
                        new Column("pid", ColumnType.Int64),
                        new Column("bid", ColumnType.Int64),
                        new Column("bs_banner_id", ColumnType.Int64),
                        new Column("order_id", ColumnType.Int64),
                        new Column("source", ColumnType.String),
                        new Column("metadata", ColumnType.String),
                        new Column("span_id", ColumnType.Int64),
                        new Column("data", ColumnType.String, null, null, "ZSTD(5)")
                )),
                new MergeTree(
                        String.format("toYYYYMM(%s)", dateColumn.getName()),
                        Arrays.asList(cidColumn.getName(), timestampColumn.getName(), sampleName),
                        sampleName,
                        8192));
    }

    protected void write(BsExportDataLine bsExportDataLine, ParserContext context) throws Exception {

        context.write(dateTimeFormat.parse(bsExportDataLine.getLogTime()),
                bsExportDataLine.getCid(),
                bsExportDataLine.getParId(),
                bsExportDataLine.getParType(),
                bsExportDataLine.getDataType(),
                bsExportDataLine.getPid(),
                bsExportDataLine.getBid(),
                bsExportDataLine.getBsBannerId(),
                bsExportDataLine.getOrderId(),
                bsExportDataLine.getSource(),
                bsExportDataLine.getMetadata(),
                bsExportDataLine.getSpanId(),
                bsExportDataLine.getData().toString()
        );
    }
}
