package ru.yandex.market.logshatter.parser.direct.bsexport;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.Base64;
import java.util.zip.GZIPInputStream;

import com.google.gson.*;
import com.google.gson.stream.JsonReader;

import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.direct.logformat.BsExportEssFormat;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import org.apache.commons.io.IOUtils;

import ru.yandex.market.logshatter.parser.direct.logformat.BsExportEssFormatChunked;
import ru.yandex.misc.io.RuntimeIoException;


public class BsExportEssLogParser extends BaseBsExportLogParser {

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonReader reader;
        try {
            if (!line.startsWith("{")) {
                line = decompress(line);
            }
            reader = new JsonReader(new StringReader(line));
            Gson gson = new GsonBuilder().create();
            JsonObject json = gson.fromJson(reader, JsonObject.class);
            if (isChunkedLog(json)) {
                writeChunkedData(json, context);
            } else {
                writeData(json, context);
            }
            reader.close();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage());
        }
    }

    private void writeData(JsonObject jsonObject, ParserContext context) throws Exception {
        Gson gson = new GsonBuilder().create();
        BsExportEssFormat bsexport = gson.fromJson(jsonObject, BsExportEssFormat.class);
        write(new BsExportDataLine()
                .setLogTime(bsexport.getTime())
                .setCid(bsexport.getCid())
                .setDataType(bsexport.getType())
                .setPid(bsexport.getPid())
                .setBid(bsexport.getBid())
                .setBsBannerId(bsexport.getBsBannerId())
                .setOrderId(bsexport.getOrderId())
                .setSpanId(bsexport.getSpanId())
                .setData(bsexport.getData()),
            context);
    }

    private void writeChunkedData(JsonObject jsonObject, ParserContext context) throws Exception {
        Gson gson = new GsonBuilder().create();
        JsonObject object = gson.fromJson(jsonObject, JsonObject.class);
        String logTime = object.get("log_time").getAsString();
        Long spanId = object.get("span_id").getAsLong();
        String type = object.get("type").getAsString();
        JsonArray dataArray = object.get("data").getAsJsonArray();
        for (JsonElement data : dataArray) {
            BsExportEssFormatChunked bsexport = gson.fromJson(data, BsExportEssFormatChunked.class);
            write(new BsExportDataLine()
                    .setLogTime(logTime)
                    .setCid(bsexport.getCid())
                    .setDataType(type)
                    .setPid(bsexport.getPid())
                    .setBid(bsexport.getBid())
                    .setBsBannerId(bsexport.getBsBannerId())
                    .setOrderId(bsexport.getOrderId())
                    .setSpanId(spanId)
                    .setData(bsexport.getData()),
                context);
        }
    }

    /**
     * На время перехода с негруппированного лога на группированный нужно делать проверку
     */
    private boolean isChunkedLog(JsonObject object) {
        return object.has("data") && object.get("data").isJsonArray();
    }

    private String decompress(String line) {
        byte[] gzipedData = Base64.getDecoder().decode(line);
        try (
                ByteArrayInputStream bis = new ByteArrayInputStream(gzipedData);
                GZIPInputStream gis = new GZIPInputStream(bis);
        ) {
            return IOUtils.toString(gis);
        } catch (IOException e) {
            throw new RuntimeIoException("Can't decompress data", e);
        }
    }
}
