package ru.yandex.market.logshatter.parser.front;

import ru.yandex.market.logshatter.parser.KeyValueExtractor;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.parser.front.helpers.AppErrorMetrikaContainer;
import ru.yandex.market.logshatter.parser.front.helpers.FrontParserUtils;
import ru.yandex.market.logshatter.parser.front.helpers.StackTraceProcessor;
import ru.yandex.market.logshatter.parser.front.helpers.tableDestription.MarketErrors;

import java.util.Date;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;


public class AppMetrikaErrorLogParser implements LogParser {
    private static Set<String> allowedAppEvents;
    private static Map<String, String> apiKeysToServiceNames;
    private static final AppMetrikaErrorFormatter METRIKA_FORMATTER = new AppMetrikaErrorFormatter();

    @Override
    public TableDescription getTableDescription() {
        return MarketErrors.DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        TskvSplitter splitter = new TskvSplitter(line);

        init(context);

        AppErrorMetrikaContainer container = METRIKA_FORMATTER.format(
            splitter,
            allowedAppEvents,
            apiKeysToServiceNames
        );

        if (container == null) {
            return;
        }

        container.writeToContext(context);
    }

    private void init(ParserContext context) {
        if (apiKeysToServiceNames != null) {
            return;
        }

        allowedAppEvents = FrontParserUtils.parseSetFromString(context.getParam("allowedAppEvents"));
        apiKeysToServiceNames = FrontParserUtils.getServiceNames(context.getParam("apiKeysToServiceNames"));
    }

    static class AppMetrikaErrorFormatter {
        private static final String FIELD_TIMESTAMP = "EventTimestamp";
        private static final String FIELD_CODE = "EventName";
        private static final String FIELD_STACK_TRACE = "EventValueCrash";
        private static final String FIELD_APP_PLATFORM = "AppPlatform";
        private static final String FIELD_APP_VERSION_NAME = "AppVersionName";
        private static final String FIELD_API_KEY = "APIKey";
        private static final String FIELD_EVENT_TYPE = "EventType";

        public AppErrorMetrikaContainer format(TskvSplitter splitter,
                                               Set<String> allowedEventTypes,
                                               Map<String, String> apiKeysToServiceNames) throws ParserException {

            String apiKey = splitter.getOptionalString(FIELD_API_KEY, "");

            boolean validSplitter = processSplitter(splitter, allowedEventTypes, apiKeysToServiceNames, apiKey);

            if (!validSplitter) {
                return null;
            }

            String service = apiKeysToServiceNames.get(apiKey);

            return buildContainer(splitter, service);
        }

        AppErrorMetrikaContainer buildContainer(TskvSplitter splitter, String service) throws ParserException {
            AppErrorMetrikaContainer container = new AppErrorMetrikaContainer();

            String appPlatform = splitter.getOptionalString(FIELD_APP_PLATFORM, "");
            String stackTrace = splitter.getOptionalString(FIELD_STACK_TRACE, "");
            String stackTraceHash = ParseUtils.sipHash64(stackTrace).toString();
            KeyValueExtractor stackTraceData = StackTraceProcessor.processStackTrace(stackTrace, appPlatform);
            long timestampSeconds = splitter.getLong(FIELD_TIMESTAMP);

            container.setDate(new Date(TimeUnit.SECONDS.toMillis(timestampSeconds)));
            container.setService(service);
            container.setStackTrace(stackTrace);
            container.setStackTraceHash(stackTraceHash);
            container.setRevision(splitter.getOptionalString(FIELD_APP_VERSION_NAME, ""));
            container.setPlatform(splitter.getOptionalString(FIELD_APP_PLATFORM, ""));
            container.setLevel("critical");

            String codeFromStackTrace = stackTraceData.getOptionalString("code", "");

            if (codeFromStackTrace.isEmpty()) {
                container.setCode(splitter.getOptionalString(FIELD_CODE, ""));
            } else {
                container.setCode(codeFromStackTrace);
            }

            container.setMessage(stackTraceData.getOptionalString("message", ""));
            container.setFile(stackTraceData.getOptionalString("file", ""));
            container.setLineNumber(stackTraceData.getOptionalInt("line", 0));

            return container;
        }

        boolean processSplitter(TskvSplitter splitter,
                                Set<String> allowedEventTypes,
                                Map<String, String> apiKeysToServiceNames,
                                String apiKey) {
            String eventType = splitter.getOptionalString(FIELD_EVENT_TYPE, "");

            if (eventType.isEmpty()) {
                return false;
            }

            if (allowedEventTypes.size() > 0 && !allowedEventTypes.contains(eventType)) {
                return false;
            }

            if (apiKey.isEmpty()) {
                return false;
            }

            return apiKeysToServiceNames.containsKey(apiKey);
        }
    }
}
