package ru.yandex.market.logshatter.parser.front;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.parser.front.helpers.FrontParserUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MetrikaTskvLogParser implements LogParser {
    private static final String COLUMN_SERVICE = "service";
    private static final String COLUMN_HOST = "host";
    private static final String COLUMN_DURATION = "duration_ms";
    private static final String COLUMN_REQ_ID = "request_id";
    private static final String COLUMN_PAGE_ID = "page_id";
    private static final String COLUMN_PLATFORM = "platform";
    private static final String COLUMN_REGION = "region";
    private static final String COLUMN_NAME = "name";
    private static final String COLUMN_PORTION = "portion";
    private static final String COLUMN_INFO_KEYS = "info_keys";
    private static final String COLUMN_INFO_VALUES = "info_values";
    private static final String COLUMN_DEVICE_MODEL = "device_model";
    private static final String COLUMN_APP_VERSION = "app_version";
    private static final String COLUMN_APP_BUILD = "app_build";
    private static final String COLUMN_TIMESTAMP_MS = "timestamp_ms";
    private static final String COLUMN_START_TIME = "start_time";
    private static final String COLUMN_START_TIME_MS = "start_time_ms";
    private static final String COLUMN_USER_AGENT = "user_agent";
    private static final String COLUMN_CLIENT_IP = "client_ip";
    private static final String COLUMN_DEVICE_ID = "device_id";
    private static final String COLUMN_VHOST = "vhost";

    private static final String ALLOWED_APP_EVENT_VALUE_TYPE = "METRIKA";
    private static final Pattern GOAL_TIMERS = Pattern.compile("^goal://(.+)/TIMERS$");
    private static Set<String> allowedAppEvents;
    private static Map<String, String> apiKeysToServiceNames;
    private Float maxAgeDays;

    public static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList(COLUMN_SERVICE, COLUMN_PLATFORM),
        new Column(COLUMN_REQ_ID, ColumnType.String),
        new Column(COLUMN_SERVICE, ColumnType.String),
        new Column(COLUMN_PLATFORM, ColumnType.String),
        new Column(COLUMN_REGION, ColumnType.UInt32),
        new Column(COLUMN_HOST, ColumnType.String),
        new Column(COLUMN_NAME, ColumnType.String),
        new Column(COLUMN_PORTION, ColumnType.String),
        new Column(COLUMN_DURATION, ColumnType.UInt32),
        new Column(COLUMN_PAGE_ID, ColumnType.String),
        new Column(COLUMN_INFO_KEYS, ColumnType.ArrayString),
        new Column(COLUMN_INFO_VALUES, ColumnType.ArrayString),
        new Column(COLUMN_DEVICE_MODEL, ColumnType.String),
        new Column(COLUMN_APP_VERSION, ColumnType.String),
        new Column(COLUMN_APP_BUILD, ColumnType.String),
        new Column(COLUMN_TIMESTAMP_MS, ColumnType.UInt64),
        new Column(COLUMN_START_TIME, ColumnType.UInt32),
        new Column(COLUMN_START_TIME_MS, ColumnType.UInt64),
        new Column(COLUMN_USER_AGENT, ColumnType.String),
        new Column(COLUMN_CLIENT_IP, ColumnType.String),
        new Column(COLUMN_DEVICE_ID, ColumnType.String),
        new Column(COLUMN_VHOST, ColumnType.String)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        init(context);

        TskvSplitter splitter = new TskvSplitter(line);

        MetrikaFormatter metrikaFormatter = MetrikaFormatterFactory.create(
            splitter,
            allowedAppEvents,
            apiKeysToServiceNames,
            maxAgeDays
        );

        List<MetrikaContainer> metrikaContainers = metrikaFormatter.format(splitter);

        if (metrikaContainers == null) {
            return;
        }

        for (MetrikaContainer metrikaContainer: metrikaContainers) {
            Object[] infoKeys;
            Object[] infoValues;
            if (metrikaContainer.getInfoKeys() == null) {
                infoKeys = infoValues = new Object[]{};
            } else {
                infoKeys = metrikaContainer.getInfoKeys().toArray();
                infoValues = metrikaContainer.getInfoValues().toArray();
            }

            context.write(
                metrikaContainer.getDate(),
                metrikaContainer.getReqId(),
                metrikaContainer.getService(),
                metrikaContainer.getPlatform(),
                metrikaContainer.getRegion(),
                context.getHost(),
                metrikaContainer.getName(),
                metrikaContainer.getPortion(),
                metrikaContainer.getDuration(),
                metrikaContainer.getPageId(),
                infoKeys,
                infoValues,
                metrikaContainer.getDeviceModel(),
                metrikaContainer.getAppVersion(),
                metrikaContainer.getAppBuild(),
                metrikaContainer.getTimestampMs(),
                metrikaContainer.getStartTime(),
                metrikaContainer.getStartTimeMs(),
                metrikaContainer.getUserAgent(),
                metrikaContainer.getClientIp(),
                metrikaContainer.getDeviceId(),
                metrikaContainer.getVhost()
            );
        }
    }

    private void init(ParserContext context) {
        String maxAgeDaysParam = context.getParam("maxAgeDays");

        if (maxAgeDaysParam != null) {
            maxAgeDays = Float.parseFloat(maxAgeDaysParam);
        }

        if (apiKeysToServiceNames != null) {
            return;
        }

        allowedAppEvents = FrontParserUtils.parseSetFromString(context.getParam("allowedAppEvents"));
        apiKeysToServiceNames = FrontParserUtils.getServiceNames(context.getParam("apiKeysToServiceNames"));
    }

    enum LogFormat {
        APP_METRIKA("metrika-mobile-log"),
        WEB_METRIKA("bs-watch-log"),
        UNKNOWN("unknown");

        private final String value;

        LogFormat(String value) {
            this.value = value;
        }

        public static LogFormat detect(TskvSplitter splitter) {
            String format = splitter.getOptionalString("tskv_format", "");

            return Arrays.stream(LogFormat.values())
                .filter(logFormat -> logFormat.value.equalsIgnoreCase(format))
                .findFirst()
                .orElse(LogFormat.UNKNOWN);
        }
    }

    enum Platform {
        ANDROID("android"),
        IOS("iOS"),
        WEB("web"),
        UNKNOWN("unknown");

        private final String name;

        Platform(String name) {
            this.name = name;
        }

        public static Platform fromString(String name) {
            for (Platform platform : Platform.values()) {
                if (platform.name.equalsIgnoreCase(name)) {
                    return platform;
                }
            }
            return null;
        }
    }

    static class MetrikaFormatterFactory {
        public static MetrikaFormatter create(TskvSplitter splitter, Set<String> allowedAppEvents,
                                              Map<String, String> apiKeysToServiceNames, Float maxAgeDays) {
            LogFormat logFormat = LogFormat.detect(splitter);

            switch (logFormat) {
                case APP_METRIKA:
                    return new AppTskvMetricaFormatter(
                        allowedAppEvents,
                        ALLOWED_APP_EVENT_VALUE_TYPE,
                        apiKeysToServiceNames,
                        maxAgeDays
                    );
                case WEB_METRIKA:
                    return new TskvMetricaFormatter(maxAgeDays);
                case UNKNOWN:
                default:
                    throw new RuntimeException(String.format("Unsupported format (%s)", logFormat));
            }
        }
    }

    interface MetrikaFormatter {
        List<MetrikaContainer> format(TskvSplitter splitter) throws ParserException;
    }

    /**
     * Описание полей:
     * tskv_format=metrika-mobile-log
     * EventValue - главное тело события, объект в формате json
     * EventValue.info - рандомная всячина, полезная для анализа события
     * EventValue.name - название порции метрики
     * EventValue.startTime - время начала события, unix ts
     * EventValue.startTimeInMs - время начала цепочки событий, unix ts в миллисекундах
     * EventValue.timestampInMs - время начала события, unix ts в миллисекундах
     * EventValue.duration - длительность события в мс
     * EventValue.requestId - айди трассировки запроса, к которму привязано событие, обязательное,
     * может быть сгенеренное
     * EventValue.date - время записи события?
     * RegionID - регион пользователя
     * Model - человекочитаемое название модели телефона
     */
    static class AppTskvMetricaFormatter extends AbstractMetrikaFormatter implements MetrikaFormatter {
        private static final String FIELD_JSON = "EventValue";
        private static final String FIELD_API_KEY = "APIKey";
        private static final String FIELD_REGION_ID = "RegionID";
        private static final String FIELD_PLATFORM = "AppPlatform";
        private static final String FIELD_EVENT_TYPE = "EventType";
        private static final String FIELD_EVENT_VALUE_TYPE = "type";
        private static final String FIELD_EVENT_VALUE_NAME = "name";
        private static final String FIELD_EVENT_PORTION = "portion";
        private static final String FIELD_EVENT_DURATION = "duration";
        private static final String FIELD_EVENT_START_TIME = "startTime";
        private static final String FIELD_EVENT_START_TIME_MS = "startTimeInMs";
        private static final String FIELD_EVENT_TIMESTAMP = "timestamp";
        private static final String FIELD_EVENT_TIMESTAMP_MS = "timestampInMs";
        private static final String FIELD_EVENT_REQUEST_ID = "requestId";
        private static final String FIELD_EVENT_INFO = "info";
        private static final String FIELD_MODEL = "Model";
        private static final String FIELD_APP_VERSION = "AppVersionName";
        private static final String FIELD_APP_BUILD = "AppBuildNumber";
        private static final String FIELD_CLIENT_IP = "ClientIP";
        private static final String FIELD_DEVICE_ID = "DeviceID";

        private final Set<String> allowedEventTypes;
        private final Map<String, String> apiKeysToServiceNames;
        private final Float maxAgeDays;
        private final String allowedAppEventValueType;
        private static final int DEFAULT_REGION_ID = -1;
        private String apiKey;

        private AppTskvMetricaFormatter(Set<String> allowedEventTypes, String allowedAppEventValueType,
            Map<String, String> apiKeysToServiceNames, Float maxAgeDays) {
            this.allowedEventTypes = allowedEventTypes;
            this.allowedAppEventValueType = allowedAppEventValueType;
            this.apiKeysToServiceNames = apiKeysToServiceNames;
            this.maxAgeDays = maxAgeDays;
        }

        Boolean processSplitter(TskvSplitter splitter) {
            String eventType = splitter.getOptionalString(FIELD_EVENT_TYPE, "");
            if (eventType.isEmpty()) {
                return false;
            }

            apiKey = splitter.getOptionalString(FIELD_API_KEY, "");
            if (apiKey.isEmpty()) {
                return false;
            }

            if (!apiKeysToServiceNames.containsKey(apiKey)) {
                return false;
            }

            if (allowedEventTypes.size() > 0 && !allowedEventTypes.contains(eventType)) {
                return false;
            }

            return true;
        }


        @Override
        String getJsonFieldName() {
            return FIELD_JSON;
        }

        @Override
        Float getMaxAgeDays() {
            return maxAgeDays;
        }

        @Override
        MetrikaContainer getSingleContainer(TskvSplitter splitter, JsonObject jsonObject) {
            // По историческим причинам этот парсер должен уметь как в TYPE === "Metrika", так и в отсутствующий TYPE
            boolean isEventValueTypeAllowed = !jsonObject.has(FIELD_EVENT_VALUE_TYPE) ||
                allowedAppEventValueType.equals(
                    jsonObject.get(FIELD_EVENT_VALUE_TYPE).getAsString()
                );

            if (
                !jsonObject.has(FIELD_EVENT_VALUE_NAME) ||
                    !jsonObject.has(FIELD_EVENT_DURATION) ||
                    !jsonObject.has(FIELD_EVENT_START_TIME) ||
                    !jsonObject.has(FIELD_EVENT_TIMESTAMP) ||
                    !jsonObject.has(FIELD_EVENT_REQUEST_ID) ||
                    !isEventValueTypeAllowed
                ) {
                return null;
            }

            MetrikaContainer metrika = new MetrikaContainer();

            metrika.setName(jsonObject.get(FIELD_EVENT_VALUE_NAME).getAsString());
            // native app metrika only has a portion, conveniently named "name"
            if (jsonObject.has(FIELD_EVENT_PORTION)) {
                metrika.setPortion(jsonObject.get(FIELD_EVENT_PORTION).getAsString());
            }

            metrika.setService(apiKeysToServiceNames.get(apiKey));

            Long timestampInMs;
            if (jsonObject.has(FIELD_EVENT_TIMESTAMP_MS)) {
                timestampInMs = jsonObject.get(FIELD_EVENT_TIMESTAMP_MS).getAsLong();
            } else {
                // Workaround for timestamp sended in ms, instead of seconds
                long timestamp = jsonObject.get(FIELD_EVENT_TIMESTAMP).getAsLong();

                if (String.valueOf(timestamp).length() <= 10) {
                    timestamp *= 1000;
                }

                timestampInMs = timestamp;
            }

            long startTimeMs;
            int startTime;
            if (jsonObject.has(FIELD_EVENT_START_TIME_MS)) {
                startTimeMs = jsonObject.get(FIELD_EVENT_START_TIME_MS).getAsLong();
            } else {
                startTimeMs = jsonObject.get(FIELD_EVENT_START_TIME).getAsLong();
            }

            // Workaround for start time sended in ms, instead of seconds
            if (String.valueOf(startTimeMs).length() <= 10) {
                startTime = (int) startTimeMs;
                startTimeMs = TimeUnit.SECONDS.toMillis(startTimeMs);
            } else {
                startTime = (int) (startTimeMs / 1000);
            }

            if (!validateTimestamps(timestampInMs, startTime, startTimeMs)) {
                return null;
            }

            metrika.setDate(new Date(timestampInMs));
            metrika.setTimestampMs(timestampInMs);
            metrika.setStartTime(startTime);
            metrika.setStartTimeMs(startTimeMs);

            metrika.setDuration(jsonObject.get(FIELD_EVENT_DURATION).getAsInt());

            metrika.setClientIp(splitter.getOptionalString(FIELD_CLIENT_IP, ""));
            metrika.setRegion(splitter.getOptionalInt(FIELD_REGION_ID, DEFAULT_REGION_ID));
            metrika.setReqId(jsonObject.get(FIELD_EVENT_REQUEST_ID).getAsString());
            metrika.setDeviceModel(splitter.getOptionalString(FIELD_MODEL, ""));
            metrika.setAppVersion(splitter.getOptionalString(FIELD_APP_VERSION, ""));
            metrika.setAppBuild(splitter.getOptionalString(FIELD_APP_BUILD, ""));
            metrika.setDeviceId(splitter.getOptionalString(FIELD_DEVICE_ID, ""));

            String platformString = splitter.getOptionalString(FIELD_PLATFORM, "");
            if (!platformString.isEmpty()) {
                Platform platform = Platform.fromString(platformString);
                if (platform != null) {
                    metrika.setPlatform(platform.name);
                }
            }

            if (jsonObject.has(FIELD_EVENT_INFO)) {
                setInfo(jsonObject, metrika);
            }

            return metrika;
        }
    }

    /**
     * regionid
     * tskv_format=bs-watch-log
     * params.requestId
     * params.info.pageId
     * params.name
     * params.startTime
     * params.duration
     * params.portion
     */
    static class TskvMetricaFormatter extends AbstractMetrikaFormatter implements MetrikaFormatter {
        private static final String FIELD_JSON = "params";
        private static final String FIELD_PARAMS_NAME = "name";
        private static final String FIELD_PARAMS_PORTION = "portion";
        private static final String FIELD_PARAMS_DURATION = "duration";
        private static final String FIELD_PARAMS_START_TIME = "startTime";
        private static final String FIELD_PARAMS_START_TIME_MS = "startTimeInMs";
        private static final String FIELD_PARAMS_TIMESTAMP_MS = "timestampInMs";
        private static final String FIELD_PARAMS_REQUEST_ID = "requestId";
        private static final String FIELD_REGION_ID = "regionid";
        private static final String FIELD_USER_AGENT = "useragent";
        private static final String FIELD_CLIENT_IP = "clientip";
        private static final String FIELD_CLIENT_IP6 = "clientip6";
        private static final String FIELD_URL = "url";
        private static final String FIELD_DEVICE_ID = "uniqid";

        private static final int DEFAULT_REGION_ID = -1;
        private final Float maxAgeDays;
        private String vhost;

        TskvMetricaFormatter(Float maxAgeDays) {
            this.maxAgeDays = maxAgeDays;
        }

        Boolean processSplitter(TskvSplitter splitter) {
            vhost = "";

            String url = splitter.getOptionalString(FIELD_URL, "");
            if (url.isEmpty()) {
                return false;
            }

            Matcher goalMatcher = GOAL_TIMERS.matcher(url);
            if (!goalMatcher.find()) {
                return false;
            }

            vhost = goalMatcher.group(1);

            return true;
        }

        @Override
        MetrikaContainer getSingleContainer(TskvSplitter splitter, JsonObject jsonObject) {
            if (
                !jsonObject.has(FIELD_PARAMS_NAME) ||
                    !jsonObject.has(FIELD_PARAMS_PORTION) ||
                    !jsonObject.has(FIELD_PARAMS_DURATION) ||
                    !jsonObject.has(FIELD_PARAMS_START_TIME) ||
                    !jsonObject.has(FIELD_PARAMS_START_TIME_MS) ||
                    !jsonObject.has(FIELD_PARAMS_REQUEST_ID) ||
                    !jsonObject.has(FIELD_PARAMS_TIMESTAMP_MS)
                ) {
                return null;
            }

            Long timestampInMs = jsonObject.get(FIELD_PARAMS_TIMESTAMP_MS).getAsLong();
            Integer startTime = jsonObject.get(FIELD_PARAMS_START_TIME).getAsInt();
            Long startTimeMs = jsonObject.get(FIELD_PARAMS_START_TIME_MS).getAsLong();

            if (!validateTimestamps(timestampInMs, startTime, startTimeMs)) {
                return null;
            }

            MetrikaContainer metrika = new MetrikaContainer();

            metrika.setDate(new Date(timestampInMs));
            metrika.setTimestampMs(timestampInMs);
            metrika.setStartTime(startTime);
            metrika.setStartTimeMs(startTimeMs);

            metrika.setClientIp(
                splitter.getOptionalString(FIELD_CLIENT_IP, splitter.getOptionalString(FIELD_CLIENT_IP6, ""))
            );
            metrika.setName(jsonObject.get(FIELD_PARAMS_NAME).getAsString());
            metrika.setPortion(jsonObject.get(FIELD_PARAMS_PORTION).getAsString());
            metrika.setDuration(jsonObject.get(FIELD_PARAMS_DURATION).getAsInt());

            metrika.setRegion(splitter.getOptionalInt(FIELD_REGION_ID, DEFAULT_REGION_ID));
            metrika.setReqId(jsonObject.get(FIELD_PARAMS_REQUEST_ID).getAsString());
            metrika.setPlatform(Platform.WEB.name);
            metrika.setUserAgent(splitter.getOptionalString(FIELD_USER_AGENT, ""));
            metrika.setDeviceId(splitter.getOptionalString(FIELD_DEVICE_ID, ""));
            metrika.setVhost(vhost);

            if (jsonObject.has(FIELD_PARAMS_INFO)) {
                setInfo(jsonObject, metrika);
            }

            return metrika;
        }

        String getJsonFieldName() {
            return FIELD_JSON;
        }

        @Override
        Float getMaxAgeDays() {
            return maxAgeDays;
        }
    }

    abstract static class AbstractMetrikaFormatter implements MetrikaFormatter {
        static final String FIELD_PARAMS_INFO = "info";
        private static final String FIELD_PARAMS_INFO_PAGE_ID = "pageId";
        private static final String FIELD_PARAMS_INFO_SERVICE = "serviceId";
        private static Set<String> extractedFields = new HashSet<>(Arrays.asList(FIELD_PARAMS_INFO_PAGE_ID,
            FIELD_PARAMS_INFO_SERVICE));

        abstract Boolean processSplitter(TskvSplitter splitter);
        abstract MetrikaContainer getSingleContainer(TskvSplitter splitter, JsonObject jsonObject);
        abstract String getJsonFieldName();
        abstract Float getMaxAgeDays();

        public List<MetrikaContainer> format(TskvSplitter splitter) {
            Boolean validSplitter = processSplitter(splitter);

            if (!validSplitter) {
                return null;
            }

            String eventValue = splitter.getOptionalString(getJsonFieldName(), "");
            if (eventValue.isEmpty()) {
                return null;
            }

            // cheap non-json filter
            if (!eventValue.startsWith("{") && !eventValue.startsWith("[")) {
                return null;
            }

            // unescaping double backslash
            eventValue = eventValue.replaceAll("\\\\", "");

            JsonParser jsonParser = new JsonParser();
            JsonElement jsonElement = jsonParser.parse(eventValue);

            ArrayList<JsonObject> allElements = new ArrayList<>();

            if (jsonElement.isJsonArray()) {
                JsonArray jsonArray = jsonElement.getAsJsonArray();

                for (JsonElement childElement: jsonArray) {
                    if (childElement.isJsonObject()) {
                        allElements.add(childElement.getAsJsonObject());
                    }
                }
            } else if (jsonElement.isJsonObject()) {
                allElements.add(jsonElement.getAsJsonObject());
            } else {
                return null;
            }

            ArrayList<MetrikaContainer> metrikaContainers = new ArrayList<>();

            for (JsonObject jsonObject: allElements) {
                MetrikaContainer metrika = getSingleContainer(splitter, jsonObject);

                if (metrika != null) {
                    metrikaContainers.add(metrika);
                }
            }

            return metrikaContainers;
        }

        Boolean validateTimestamps(Long timestampInMs, Integer startTime, Long startTimeInMs) {
            Float maxAgeDays = getMaxAgeDays();

            return
                timestampInMs != 0 &&
                timestampInMs <= System.currentTimeMillis() && (
                    maxAgeDays == null ||
                    timestampInMs > System.currentTimeMillis() - maxAgeDays * TimeUnit.DAYS.toMillis(1)
                ) &&
                startTime <= System.currentTimeMillis() / 1000 &&
                startTimeInMs <= System.currentTimeMillis();
        }

        void setInfo(JsonObject jsonObject, MetrikaContainer metrika) {
            JsonObject infoObject = jsonObject.getAsJsonObject(FIELD_PARAMS_INFO);

            ArrayList<String> infoKeys = new ArrayList<>();
            ArrayList<String> infoValues = new ArrayList<>();

            if (infoObject.has(FIELD_PARAMS_INFO_PAGE_ID)) {
                metrika.setPageId(infoObject.get(FIELD_PARAMS_INFO_PAGE_ID).getAsString());
            }

            if (infoObject.has(FIELD_PARAMS_INFO_SERVICE)) {
                metrika.setService(infoObject.get(FIELD_PARAMS_INFO_SERVICE).getAsString());
            }

            for (Map.Entry<String, JsonElement> entry : infoObject.entrySet()) {
                if (extractedFields.contains(entry.getKey())) {
                    continue;
                }

                if (!entry.getValue().isJsonPrimitive()) {
                    continue;
                }

                infoKeys.add(entry.getKey());
                infoValues.add(entry.getValue().getAsString());
            }

            metrika.setInfoKeys(infoKeys);
            metrika.setInfoValues(infoValues);
        }
    }

    static class MetrikaContainer {
        private String service = "";
        private String reqId = "";
        private String platform = "";
        private int region;
        private String name = "";
        private String portion = "";
        private int duration;
        private int startTime;
        private long startTimeMs;
        private long timestampMs;
        private String pageId = "";
        private Date date;
        private String deviceModel = "";
        private String appVersion = "";
        private String appBuild = "";
        private String userAgent = "";
        private List<String> infoKeys;
        private List<String> infoValues;
        private String clientIp = "";
        private String deviceId = "";
        private String vhost = "";


        public String getService() {
            return service;
        }

        public void setService(String service) {
            this.service = service;
        }

        public String getReqId() {
            return this.reqId;
        }

        public void setReqId(String reqId) {
            this.reqId = reqId;
        }

        public String getPlatform() {
            return platform;
        }

        public void setPlatform(String platform) {
            this.platform = platform;
        }

        public int getRegion() {
            return region;
        }

        public void setRegion(int region) {
            this.region = region;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getPortion() {
            return portion;
        }

        public void setPortion(String portion) {
            this.portion = portion;
        }

        public int getDuration() {
            return duration;
        }

        public void setDuration(int duration) {
            this.duration = duration;
        }

        public String getPageId() {
            return pageId;
        }

        public void setPageId(String pageId) {
            this.pageId = pageId;
        }

        public Date getDate() {
            return date;
        }

        public void setDate(Date date) {
            this.date = date;
        }

        public String getDeviceModel() {
            return deviceModel;
        }

        public void setDeviceModel(String model) {
            this.deviceModel = model;
        }

        public String getAppVersion() {
            return appVersion;
        }

        public void setAppVersion(String version) {
            this.appVersion = version;
        }

        public String getAppBuild() {
            return this.appBuild;
        }

        public void setAppBuild(String build) {
            this.appBuild = build;
        }

        public List<String> getInfoKeys() {
            return infoKeys;
        }

        public void setInfoKeys(List<String> infoKeys) {
            this.infoKeys = infoKeys;
        }

        public List<String> getInfoValues() {
            return infoValues;
        }

        public void setInfoValues(List<String> infoValues) {
            this.infoValues = infoValues;
        }

        public String getUserAgent() {
            return userAgent;
        }

        public void setUserAgent(String userAgent) {
            this.userAgent = userAgent;
        }

        public long getTimestampMs() {
            return timestampMs;
        }

        public void setTimestampMs(long timestampMs) {
            this.timestampMs = timestampMs;
        }

        public int getStartTime() {
            return startTime;
        }

        public void setStartTime(int startTime) {
            this.startTime = startTime;
        }

        public long getStartTimeMs() {
            return startTimeMs;
        }

        public void setStartTimeMs(long startTimeMs) {
            this.startTimeMs = startTimeMs;
        }

        public String getClientIp() {
            return clientIp;
        }

        public void setClientIp(String clientIp) {
            this.clientIp = clientIp;
        }

        public String getDeviceId() {
            return deviceId;
        }

        public void setDeviceId(String deviceId) {
            this.deviceId = deviceId;
        }

        public String getVhost() {
            return vhost;
        }

        public void setVhost(String vhost) {
            this.vhost = vhost;
        }
    }
}
