package ru.yandex.market.logshatter.parser.front;

import ru.yandex.common.util.StringUtils;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.trace.Environment;

import java.nio.file.Path;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 31/03/15
 */
public class NodeJsErrorsLogParser implements LogParser {
    public static final int MIN_COLUMN_LENGTH = 6;
    public static final int EXPERIMENT_ID_DIR_INDEX = 4;
    public static final int EXPERIMENT_SUFFIXED_DIR_INDEX = 3;
    public static final String EXPERIMENT_SUFFIX = "-exp";
    public static final List<String> allowedSeverity = Arrays.asList("error", "warn");
    public static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("component", ColumnType.String),
        new Column("error_code", ColumnType.String),
        new Column("resource", ColumnType.String),
        new Column("resource_method", ColumnType.String),
        new Column("resource_status", ColumnType.Int16, "-1"),
        new Column("exp_id", ColumnType.String),
        new Column("severity", ColumnType.String, "'error'"),
        new Column("request_id", ColumnType.String),
        new Column("environment", EnumColumnType.enum8(Environment.class), "'UNKNOWN'"),
        new Column("instance_id", ColumnType.UInt32, "0")
    );

    private final DateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss Z");
    private final String RE_DOT = "\\.";
    private final String RE_DIGITS = "^\\d+$";
    private Pattern pattern = Pattern.compile(RE_DIGITS);
    private Matcher matcher = pattern.matcher("");
    private EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    public static String extractExperimentId(Path path) {
        if (path == null
            || path.getNameCount() <= EXPERIMENT_SUFFIXED_DIR_INDEX
            || path.getNameCount() <= EXPERIMENT_ID_DIR_INDEX
            || !path.getName(EXPERIMENT_SUFFIXED_DIR_INDEX).getFileName().toString().endsWith(EXPERIMENT_SUFFIX)) {
            return StringUtils.EMPTY;
        }

        return path.getName(EXPERIMENT_ID_DIR_INDEX).getFileName().toString();
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        String[] splits = line.split("\t");
        if (splits.length < MIN_COLUMN_LENGTH) {
            return;
        }

        Date date = dateFormat.parse(splits[0]);

        String severity = splits[2].toLowerCase();
        if (!allowedSeverity.contains(severity)) {
            return;
        }

        String component = splits[3];
        if (!component.equals("APP") && !component.equals("STOUT") && !component.equals("RESOURCE")) {
            return;
        }

        String errorCode = splits[4];

        String requestId = splits[5];
        if (requestId.equals("-")) {
            requestId = "";
        }

        String resource = "";
        String resourceMethod = "";
        Integer resourceStatus = -1;
        if (component.equals("RESOURCE")) {
            String[] resourceSplits = splits[5].split(RE_DOT);
            String resourceStatusString = splits[6];

            if (resourceSplits.length >= 2) {
                resource = resourceSplits[0];
                resourceMethod = resourceSplits[1];
            }
            if (matcher.reset(resourceStatusString).matches()) {
                resourceStatus = Integer.valueOf(resourceStatusString);
            }

            requestId = splits[8];
            if (requestId.equals("-")) {
                requestId = "";
            }
        }

        String expId = extractExperimentId(context.getFile());

        Environment environment = environmentMapper.getEnvironment(context);

        context.write(
            date,
            context.getHost(),
            component,
            errorCode,
            resource,
            resourceMethod,
            resourceStatus,
            expId,
            severity,
            requestId,
            environment,
            context.getInstanceId()
        );
    }
}
