package ru.yandex.market.logshatter.parser.front;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.parser.trace.Environment;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.Date;

public class TimersTskvLogParser implements LogParser {
    private static final String COLUMN_SERVICE = "service";
    private static final String COLUMN_ENVIRONMENT = "environment";
    private static final String COLUMN_HOST = "host";
    private static final String COLUMN_LABEL = "label";
    private static final String COLUMN_TYPE = "type";
    private static final String COLUMN_DURATION = "duration_ms";
    private static final String COLUMN_REQ_ID = "request_id";
    private static final String COLUMN_PAGE_ID = "page_id";
    private static final String FILE_SUFFIX = "-timers.log";

    public static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList(COLUMN_SERVICE, COLUMN_ENVIRONMENT),
        new Column(COLUMN_SERVICE, ColumnType.String),
        new Column(COLUMN_HOST, ColumnType.String),
        new Column(COLUMN_LABEL, ColumnType.String),
        new Column(COLUMN_TYPE, ColumnType.String),
        new Column(COLUMN_DURATION, ColumnType.UInt16),
        new Column(COLUMN_REQ_ID, ColumnType.String),
        new Column(
            COLUMN_ENVIRONMENT, EnumColumnType.enum8(Environment.class),
            String.format("'%s'", Environment.UNKNOWN.toString())
        ),
        new Column(COLUMN_PAGE_ID, ColumnType.String)
    );

    private EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        TskvSplitter splitter = new TskvSplitter(line);

        context.write(new Date(splitter.getLong("timestamp")),
            getServiceName(context.getFile()),
            context.getHost(),
            splitter.getString(COLUMN_LABEL),
            splitter.getString(COLUMN_TYPE),
            splitter.getInt(COLUMN_DURATION),
            splitter.getString(COLUMN_REQ_ID),
            environmentMapper.getEnvironment(context),
            splitter.getOptionalString(COLUMN_PAGE_ID, "")
        );
    }

    private String getServiceName(Path file) {
        String fileName = file.getFileName().toString();
        if (!fileName.endsWith(FILE_SUFFIX)) {
            return "";
        }

        return fileName.substring(0, fileName.length() - FILE_SUFFIX.length());
    }
}
