package ru.yandex.market.logshatter.parser.front.errorBooster.qloudNode;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.front.errorBooster.schema.Schema;
import ru.yandex.market.logshatter.useragent.UserAgentDetector;

import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Date;

public class ErrorsParser implements LogParser {

    public ErrorsParser() {}

    private static final Logger log = LogManager.getLogger();

    @Override
    public TableDescription getTableDescription() {
        return Schema.TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonParser jsonParser = new JsonParser();

        JsonObject jsonObject;
        try {
            JsonElement jsonElement = jsonParser.parse(line);

            if (!jsonElement.isJsonObject()) {
                return;
            }

            jsonObject = jsonElement.getAsJsonObject();
        } catch (JsonParseException ignored) {
            return;
        }

        UserAgentDetector userAgentDetector = context.getUserAgentDetector();

        CommonContainer commonContainer = new CommonContainer(userAgentDetector);

        if (!commonContainer.fill(jsonObject)) {
            return;
        }
        ErrorsContainer errorsContainer = new ErrorsContainer();

        if (!errorsContainer.fill(jsonObject)) {
            return;
        }

        String rawDate = jsonObject.get("@timestamp").getAsString();
        ZonedDateTime date = Instant.parse(rawDate).atZone(ZoneId.of("UTC"));
        // https://st.yandex-team.ru/ERRORBOOSTER-252
        this.logBadDate(date, rawDate, errorsContainer.getClientTimestamp());

        context.write(
            date,
            ArrayUtils.addAll(commonContainer.serialize(), errorsContainer.serialize())
        );
    }

    private void logBadDate(ZonedDateTime date, String rawDate, Integer clientTimestamp) {
        if (clientTimestamp > 0) {
            Date clientDate = new Date((long) clientTimestamp * 1000);
            Long diffDays = ChronoUnit.DAYS.between(clientDate.toInstant(), date.toInstant());
            if (diffDays > 1) {
                log.info("Bad date. @timestamp: {}. Parsed date: {}. Client date: {}. Diff days: {}", rawDate, date, clientDate, diffDays);
            }
        }
    }
}
