package ru.yandex.market.logshatter.parser.front.errorBooster.redirlog;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import org.apache.commons.lang.StringUtils;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

public class RedirlogParser {

    private static final BlockstatDecoder BLOCKSTAT_DECODER = new BlockstatDecoder();

    public static Map<String, String> getParams(String line) {
        List<String> params = Splitter.on("@@").splitToList(line);
        Map<String, String> redirParams = new HashMap<>();
        for (String param : params) {
            int splitIndex = param.indexOf('=');
            if (splitIndex > 0) {
                redirParams.put(param.substring(0, splitIndex), param.substring(splitIndex + 1));
            }
        }

        /**
         * https://wiki.yandex-team.ru/data/logs/redir/
         * Последние три поля - просто значения. Это, по порядку:
         * unix timestamp
         * IP-адрес(а) пользователя (HTTP заголовки X-Forwarded-For или X-Forwarded-For-Y дописываются через запятую если присутствуют. Актуальный список заголовков лучше уточнять у Михаила Рублева)
         * yandexuid
         */
        int paramsLength = params.size();
        redirParams.put("timestamp", params.get(paramsLength - 3));
        redirParams.put("yandexuid", params.get(paramsLength - 1));

        if (!redirParams.containsKey("ruip")) {
            redirParams.put("ruip", StringUtils.substringBefore(params.get(paramsLength - 2), ","));
        }

        return redirParams;
    }

    public static Map<String, String> getVars(Map<String, String> redirParams) {
        return parseVars(redirParams, (name) -> name, (name, value) -> value);
    }

    public static Map<String, String> getVars(Map<String, String> redirParams, Collection<String> blockstatValues) {
        return parseVars(
            redirParams,
            (name) -> BLOCKSTAT_DECODER.decode(name),
            (name, value) -> blockstatValues.contains(name) ? BLOCKSTAT_DECODER.decode(value) : value
        );
    }

    private static Map<String, String> parseVars(Map<String, String> redirParams, Function<String, String> prepareName, BiFunction<String, String, String> prepareValue) {
        List<String> varsList = Splitter.on(",").splitToList(redirParams.getOrDefault("vars", ""));
        Map<String, String> vars = new HashMap<>();
        for (String var : varsList) {
            int splitIndex = var.indexOf('=');
            if (splitIndex > 0) {
                String name = var.substring(0, splitIndex);
                String value = var.substring(splitIndex + 1);

                if (name.startsWith("-")) {
                    name = name.substring(1);
                }

                name = prepareName.apply(name);

                vars.put(name, prepareValue.apply(name, value));
            }
        }

        return vars;
    }
}
