package ru.yandex.market.logshatter.parser.front.errorBooster.redirlog.errors;

import org.apache.commons.lang.StringUtils;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.front.errorBooster.LogLevel;
import ru.yandex.market.logshatter.parser.front.errorBooster.Parser;
import ru.yandex.market.logshatter.parser.front.errorBooster.Runtime;
import ru.yandex.market.logshatter.parser.front.errorBooster.schema.ErrorsSchema;

import java.net.URLDecoder;
import java.util.Map;

import static ru.yandex.market.logshatter.parser.ParseUtils.ipv4ToLong;
import static ru.yandex.market.logshatter.parser.ParseUtils.sipHash64;

public class ErrorsContainer extends ErrorsSchema {
    public ErrorsContainer() { }

    public boolean fill(Map<String, String> redirVars, Map<String, String> customVars) throws Exception {
        originalMessage = URLDecoder.decode(customVars.getOrDefault("msg", ""), "UTF-8");
        message = prepareMessage(originalMessage);

        ip = ipv4ToLong(redirVars.getOrDefault("ruip", ""));

        if (StringUtils.isEmpty(message)) {
            return false;
        }

        messageId = sipHash64(message);

        // Да, это "url", потому что это "url" до файла, из которого случилась ошибка
        file = URLDecoder.decode(customVars.getOrDefault("url", ""), "UTF-8");
        fileId = sipHash64(file);

        line = ParseUtils.parseUnsignedInt(customVars.get("line"), 0);
        col = ParseUtils.parseUnsignedInt(customVars.get("col"), 0);
        block = URLDecoder.decode(customVars.getOrDefault("block", ""), "UTF-8");
        method = URLDecoder.decode(customVars.getOrDefault("method", ""), "UTF-8");

        originalStackTrace = URLDecoder.decode(customVars.getOrDefault("stack", ""), "UTF-8");
        stackTrace = prepareStacktrace(originalStackTrace);
        stackTraceId = sipHash64(stackTrace);

        clientTimestamp = prepareClientTimestamp(customVars.get("ts"));

        userAgent = URLDecoder.decode(customVars.getOrDefault("ua", ""), "UTF-8");
        userAgentId = sipHash64(userAgent);

        String path = redirVars.get("path");
        PathError pathError = PathError.fromString(path);

        source = prepareSource(URLDecoder.decode(customVars.getOrDefault("source", ""), "UTF-8"), pathError);
        sourceMethod = URLDecoder.decode(customVars.getOrDefault("sourceMethod", ""), "UTF-8");
        sourceType = URLDecoder.decode(customVars.getOrDefault("type", ""), "UTF-8");

        logLevel = prepareLevel(customVars.getOrDefault("level", ""), pathError);

        if (pathError == PathError.NODEJS) {
            runtime = Runtime.NODEJS;
        }

        parser = Parser.REDIR_LOG;

        return true;
    }

    public String prepareMessage(String message) {
        // Script error. => Script error
        message = StringUtils.removeEnd(message, ".");

        // Uncaught TypeError => TypeError
        // Uncaught SyntaxError => SyntaxError
        // Uncaught ReferenceError => ReferenceError
        message = StringUtils.removeStart(message, "Uncaught ");
        message = replaceUrls(message, "MESSAGE");
        return message;
    }

    public String prepareStacktrace(String stackTrace) {
        stackTrace = replaceUrls(stackTrace, "STACKTRACE");
        return stackTrace;
    }

    public LogLevel prepareLevel(String level, PathError pathError) {
        LogLevel logLevel = LogLevel.fromString(level);

        if (logLevel != LogLevel.UNKNOWN) {
            return logLevel;
        }

        switch(pathError) {
            case UNCAUGHT:
            case SCRIPT:
            case EXTERNAL: return LogLevel.ERROR;
            default : return logLevel;
        }
    }

    public String prepareSource(String source, PathError pathError) {
        if (StringUtils.isNotEmpty(source)) {
            return source;
        }

        switch(pathError) {
            case CLIENT: return "client";
            case UNCAUGHT: return "uncaught";
            case SCRIPT: return "script";
            case EXTERNAL: return "external";
        }

        return source;
    }
}
