package ru.yandex.market.logshatter.parser.front.errorBooster.stackParser;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class BrowserStackParser implements StackParser {
    private static final Pattern AT_FORMAT = Pattern.compile("^\\s+at .*$");

    /**
     * at name (url:line:col)
     */
    private static final Pattern FORMAT_AT_NAME_URL_LINE_COL = Pattern.compile("^\\s+at (?<name>.+) \\((?<url>\\S+):(?<line>\\d+):(?<col>\\d+)\\)$");

    /**
     * at url:line:col
     */
    private static final Pattern FORMAT_AT_URL_LINE_COL = Pattern.compile("^\\s+at (?<url>.+):(?<line>\\d+):(?<col>\\d+)$");

    /**
     * name@url:line:col
     * name@url
     */
    private static final Pattern FORMAT_NAME_URL_LINE_COL = Pattern.compile("^(?<name>[^@]*)@(?<url>.+?)(?::(?<line>\\d+):(?<col>\\d+))?$");

    /**
     * url:line:col
     */
    private static final Pattern FORMAT_URL_LINE_COL = Pattern.compile("^(?<url>.+):(?<line>\\d+):(?<col>\\d+)$");

    private final StackFrame[] stackFrames;

    /**
     * Эвристически определяем формат стека, и парсим стек.
     *
     * @param stack стек ошибки из браузера в виде строки
     */
    public BrowserStackParser(String stack) {
        String[] lines = StringUtils.split(stack, '\n');
        List<StackFrame> foundFrames = new ArrayList<>(lines.length);
        boolean stackUsesAtFormat = AT_FORMAT.matcher(lines[lines.length - 1]).matches();

        for (String line : lines) {
            StackFrame frame = null;

            if (stackUsesAtFormat) {
                frame = StringUtils.endsWith(line, ")") ?
                    parseAtNameUrlLineColFormat(line) :
                    parseAtUrlLineColFormat(line);
            } else {
                if (StringUtils.contains(line, '@')) {
                    frame = parseNameUrlLineColFormat(line);
                }
                if (frame == null) {
                    frame = parseUrlLineColFormat(line);
                }
            }

            if (frame != null) {
                foundFrames.add(frame);
            }
        }

        stackFrames = foundFrames.toArray(new StackFrame[0]);
    }

    @VisibleForTesting
    static StackFrame parseAtNameUrlLineColFormat(String line) {
        Matcher matcher = FORMAT_AT_NAME_URL_LINE_COL.matcher(line);

        return matcher.matches() ? createFrame(
            matcher.group("name"),
            matcher.group("url"),
            matcher.group("line"),
            matcher.group("col")
        ) : null;
    }

    @VisibleForTesting
    static StackFrame parseNameUrlLineColFormat(String line) {
        Matcher matcher = FORMAT_NAME_URL_LINE_COL.matcher(line);

        return matcher.matches() ? createFrame(
            matcher.group("name"),
            matcher.group("url"),
            matcher.group("line"),
            matcher.group("col")
        ) : null;
    }

    @VisibleForTesting
    static StackFrame parseAtUrlLineColFormat(String line) {
        Matcher matcher = FORMAT_AT_URL_LINE_COL.matcher(line);

        return matcher.matches() ? createFrame(
            "(anonymous)",
            matcher.group("url"),
            matcher.group("line"),
            matcher.group("col")
        ) : null;
    }

    @VisibleForTesting
    static StackFrame parseUrlLineColFormat(String line) {
        Matcher matcher = FORMAT_URL_LINE_COL.matcher(line);

        return matcher.matches() ? createFrame(
            "(anonymous)",
            matcher.group("url"),
            matcher.group("line"),
            matcher.group("col")
        ) : null;
    }

    private static StackFrame createFrame(String name, String url, String line, String col) {
        return new StackFrame(
            StringUtils.defaultIfEmpty(name, "(anonymous)"),
            StringUtils.defaultIfEmpty(url, StringUtils.EMPTY),
            Integer.parseInt(StringUtils.defaultIfEmpty(line, "0")),
            Integer.parseInt(StringUtils.defaultIfEmpty(col, "0"))
        );
    }

    @Override
    public StackFrame[] getStackFrames() {
        return stackFrames;
    }
}
