package ru.yandex.market.logshatter.parser.front.formatters;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.lang.StringEscapeUtils;
import ru.yandex.market.logshatter.parser.KeyValueExtractor;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.parser.front.errorBooster.LogLevel;
import ru.yandex.market.logshatter.parser.front.helpers.AppErrorMetrikaContainer;
import ru.yandex.market.logshatter.parser.front.helpers.StackTraceProcessor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;


public class AppMetrikaEventFormatter {
    private static final String FIELD_TIMESTAMP = "EventTimestamp";
    private static final String FIELD_APP_PLATFORM = "AppPlatform";
    private static final String FIELD_APP_VERSION_NAME = "AppVersionName";
    private static final String FIELD_API_KEY = "APIKey";
    private static final String FIELD_EVENT_TYPE = "EventType";
    private static final String FIELD_EVENT_VALUE = "EventValue";

    private enum EventValueField {
        TYPE("type"),
        NAME("name"),
        INFO("info"),
        LEVEL("level"),
        MESSAGE("message"),
        REQUEST_ID("requestId"),
        STACK_TRACE("stackTrace"),
        PORTION("portion");

        private final String value;

        EventValueField(String value) {
            this.value = value;
        }

        public String getValue() {
            return this.value;
        }

        static public List<String> getAllValuesAsStringList() {
            return Arrays.stream(EventValueField.values())
                .map(EventValueField::getValue)
                .collect(Collectors.toList());
        }
    }

    public AppErrorMetrikaContainer format(TskvSplitter splitter, Set<String> allowedEventTypes,
                                           Map<String, String> apiKeysToServiceNames,
                                           String allowedAppEventValueType) throws ParserException {
        String apiKey = splitter.getString(FIELD_API_KEY);
        boolean validSplitter = processSplitter(splitter, allowedEventTypes, apiKeysToServiceNames, apiKey);

        if (!validSplitter) {
            return null;
        }

        String eventValue = splitter.getOptionalString(FIELD_EVENT_VALUE, "");

        // cheap non-json filter
        if (eventValue.isEmpty() || !eventValue.startsWith("{")) {
            return null;
        }

        JsonParser jsonParser = new JsonParser();
        JsonElement jsonElement = jsonParser.parse(eventValue);

        JsonObject jsonObject;

        if (jsonElement.isJsonObject()) {
            jsonObject = jsonElement.getAsJsonObject();
        } else {
            return null;
        }

        String service = apiKeysToServiceNames.get(apiKey);

        return buildContainer(splitter, service, jsonObject, allowedAppEventValueType);
    }

    private AppErrorMetrikaContainer buildContainer(TskvSplitter splitter, String service,
                                                    JsonObject eventValueJsonObject, String allowedAppEventValueType)
        throws ParserException {
        boolean isEventValueTypeAllowed = eventValueJsonObject.has(EventValueField.TYPE.getValue()) &&
            allowedAppEventValueType.equals(
                eventValueJsonObject.get(EventValueField.TYPE.getValue()).getAsString()
            );

        if (!isEventValueTypeAllowed) {
            return null;
        }

        AppErrorMetrikaContainer container = new AppErrorMetrikaContainer();

        String appPlatform = splitter.getOptionalString(FIELD_APP_PLATFORM, "");
        container.setPlatform(appPlatform);

        JsonObject infoJsonObject = null;
        if (eventValueJsonObject.has(EventValueField.INFO.getValue())) {
            infoJsonObject = eventValueJsonObject.get(EventValueField.INFO.getValue()).getAsJsonObject();
        }
        processInfo(container, infoJsonObject, appPlatform, eventValueJsonObject);

        if (eventValueJsonObject.has(EventValueField.LEVEL.getValue())) {
            String logLevelString = eventValueJsonObject.get(EventValueField.LEVEL.getValue()).getAsString();
            LogLevel logLevel = LogLevel.fromString(logLevelString.toLowerCase());

            if (logLevel != null) {
                container.setLevel(logLevel.toString().toLowerCase());
            }
        }

        if (eventValueJsonObject.has(EventValueField.NAME.getValue())) {
            container.setCode(eventValueJsonObject.get(EventValueField.NAME.getValue()).getAsString());
        }

        if (eventValueJsonObject.has(EventValueField.REQUEST_ID.getValue())) {
            container.setRequestId(eventValueJsonObject.get(EventValueField.REQUEST_ID.getValue()).getAsString());
        }

        container.setService(service);
        container.setRevision(splitter.getOptionalString(FIELD_APP_VERSION_NAME, ""));
        container.setDate(new Date(
            TimeUnit.SECONDS.toMillis(splitter.getLong(FIELD_TIMESTAMP))
        ));


        return container;
    }

    private void processInfo(
        @Nonnull final AppErrorMetrikaContainer container,
        @Nullable final JsonObject infoJsonObject,
        @Nullable final String appPlatform,
        @Nonnull final JsonObject eventValueJsonObject
    ) throws ParserException {
        if (infoJsonObject != null) {
            if (infoJsonObject.has(EventValueField.MESSAGE.getValue())) {
                container.setMessage(infoJsonObject.get(EventValueField.MESSAGE.getValue()).getAsString());
            }

            if (infoJsonObject.has(EventValueField.STACK_TRACE.getValue())) {
                String stackTrace = infoJsonObject.get(EventValueField.STACK_TRACE.getValue()).getAsString();
                container.setStackTrace(stackTrace);

                String stackTraceHash = ParseUtils.sipHash64(stackTrace).toString();
                container.setStackTraceHash(stackTraceHash);

                KeyValueExtractor stackTraceData = StackTraceProcessor.processStackTrace(
                    StringEscapeUtils.escapeJava(stackTrace), appPlatform
                );
                container.setFile(stackTraceData.getString("file"));
                container.setLineNumber(stackTraceData.getInt("line"));
            }
        }

        processExtra(container, infoJsonObject, eventValueJsonObject);
    }

    private void processExtra(
        @Nonnull final AppErrorMetrikaContainer container,
        @Nullable final JsonObject infoJsonObject,
        @Nonnull final JsonObject eventValueJsonObject
    ) {
        List<String> declaredFields = EventValueField.getAllValuesAsStringList();

        final List<String> extraKeys;
        final List<String> extraValues;
        if (infoJsonObject != null) {
            extraKeys = infoJsonObject.keySet().stream()
                .filter(field -> !declaredFields.contains(field))
                .collect(Collectors.toList());

            extraValues = extraKeys.stream()
                .map(field -> {
                    try {
                        return infoJsonObject.get(field).getAsString();
                    } catch (UnsupportedOperationException e) {
                        return "";
                    }
                })
                .collect(Collectors.toList());
        } else {
            extraKeys = new ArrayList<>();
            extraValues = new ArrayList<>();
        }

        String portion = null;
        if (eventValueJsonObject.has(EventValueField.PORTION.getValue())) {
            portion = eventValueJsonObject.get(EventValueField.PORTION.getValue()).getAsString();
        }
        if (portion != null && !portion.isEmpty()) {
            extraKeys.add(EventValueField.PORTION.getValue());
            extraValues.add(portion);
        }

        container.setExtraKeys(extraKeys.toArray(new String[]{}));
        container.setExtraValues(extraValues.toArray(new String[]{}));
    }

    private boolean processSplitter(TskvSplitter splitter, Set<String> allowedEventTypes,
                                    Map<String, String> apiKeysToServiceNames, String apiKey) {
        String eventType = splitter.getOptionalString(FIELD_EVENT_TYPE, "");
        boolean isAllowed = allowedEventTypes.size() > 0 && !allowedEventTypes.contains(eventType);

        if (apiKey.isEmpty() || eventType.isEmpty() || isAllowed) {
            return false;
        }

        return apiKeysToServiceNames.containsKey(apiKey);
    }
}
