package ru.yandex.market.logshatter.parser.front.helpers;

import ru.yandex.market.logshatter.parser.KeyValueExtractor;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class StackTraceProcessor {
    /**
     * Разбираем stacktrace java-приложения вида:
     *
     * код0: сообщение0: код: сообщение\n
     * \t  at метод(файл.java:строка)\n
     * ...
     *
     * Регулярное выражение запоминает: код (2), сообщение (4), файл.java (6), строка (7).
     */
    private static final Pattern STACK_TRACE_PATTERN_ANDROID_LONG =
        Pattern.compile("^(.+?:\\s.+?:\\s)?(.+?)(:\\s(.+?))?\\\\n\\\\tat(.+?\\((.+?):([0-9]+)\\))\\\\n\\\\tat");

    /**
     * Разбираем stacktrace java-приложения вида:
     *
     * код: сообщение; no stack trace available\n
     *
     * Регулярное выражение запоминает: код (1), сообщение (2).
     */
    private static final Pattern STACK_TRACE_PATTERN_ANDROID_SHORT =
        Pattern.compile("^(.+?):\\s(.+?); no stack trace available\\\\n$");

    /**
     * Разбираем stacktrace iOS-приложения вида: *сотни тысяч символов*
     */
    private static final Pattern STACK_TRACE_PATTERN_IOS =
        Pattern.compile("\\(\\d+\\)\\\\nCode Type:.*Exception Type:\\s*(.+)\\\\n" +
                "Exception Codes.*CrashDoctor Diagnosis:\\s*(.*)\\\\n");

    private StackTraceProcessor() {}

    public static KeyValueExtractor processStackTrace(String stackTrace, String appPlatform) {
        Map<String, String> map = new HashMap<String, String>() {{
            put("code", "");
            put("message", "");
            put("file", "");
            put("line", "0");
        }};

        KeyValueExtractor result = new KeyValueExtractor(map);

        if ("iOS".equals(appPlatform)) {
            Matcher iOSMatcher = STACK_TRACE_PATTERN_IOS.matcher(stackTrace);

            if (iOSMatcher.find()) {
                map.put("code", iOSMatcher.group(1));
                map.put("message", iOSMatcher.group(2));
            }
        }

        if ("android".equals(appPlatform)) {
            Matcher androidMatcherShort = STACK_TRACE_PATTERN_ANDROID_SHORT.matcher(stackTrace);
            Matcher androidMatcherLong = STACK_TRACE_PATTERN_ANDROID_LONG.matcher(stackTrace);

            if (androidMatcherShort.find()) {
                map.put("code", androidMatcherShort.group(1));
                map.put("message", androidMatcherShort.group(2));
            } else if (androidMatcherLong.find()) {
                map.put("code", androidMatcherLong.group(2));
                map.put("message", androidMatcherLong.group(4));
                map.put("file", androidMatcherLong.group(6));
                map.put("line", androidMatcherLong.group(7));
            }
        }

        return result;
    }
}
