package ru.yandex.market.logshatter.parser.internal;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumArrayColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.TskvSplitter;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/08/15
 * ru.yandex.market.clickphite.metric.MetricService.logQuery();
 */
public class ClickphiteMetricLogParser implements LogParser {
    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("type", ColumnType.String),
        new Column("table", ColumnType.String),
        new Column("id", ColumnType.String), // deprecated
        new Column("storage", ColumnType.String), // deprecated
        new Column("period", ColumnType.String),
        new Column("start_date", ColumnType.DateTime),
        new Column("end_date", ColumnType.DateTime),
        new Column("query_time_millis", ColumnType.Int32),
        new Column("send_time_millis", ColumnType.Int32), // deprecated
        new Column("metrics_send", ColumnType.UInt32), // deprecated
        new Column("metrics_ignored", ColumnType.UInt32), // deprecated
        new Column("rows_read", ColumnType.UInt32),
        new Column("rows_ignored", ColumnType.UInt32),
        new Column("invalid_rows_ignored_per_id", ColumnType.ArrayUInt32),
        new Column("metric_ids", ColumnType.ArrayString, "[id]"),
        new Column("storages", ColumnType.ArrayString, "[storage]"), // deprecated
        new Column("send_time_millis_per_id", ColumnType.ArrayInt32, "[send_time_millis]"),
        new Column(
            "storage_per_id", EnumArrayColumnType.enum8Array(Storage.class),
            "arrayMap(id -> if((id LIKE 'graphite%') = 1, 'GRAPHITE', 'STATFACE'), metric_ids)"
        ),
        new Column("metrics_send_per_id", ColumnType.ArrayUInt32, "[metrics_send]"),
        new Column("total_metrics_count_in_group", ColumnType.UInt16, "length(metric_ids)"),
        new Column(
            "query_weight", EnumColumnType.enum8(QueryWeight.class),
            "if(period IN ('QUARTER', 'MONTH', 'WEEK', 'DAY'), 'HEAVY', if((period = 'HOUR') OR (table = 'market.trace'), 'MEDIUM', 'LIGHT'))"
        )
    );
    private static final String[] EMPTY_STRING_ARRAY = {};

    private DateFormat dateFormat = new SimpleDateFormat("[dd/MMM/yyyy:HH:mm:ss Z]", Locale.US);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        if (line.startsWith("tskv")) {
            parseTskv(line, context);
        } else {
            parseTabSeparated(line, context);
        }
    }

    private void parseTskv(String line, ParserContext context) throws Exception {
        TskvSplitter splitter = new TskvSplitter(line);

        context.write(
            splitter.getDate("date", dateFormat),
            context.getHost(),
            "", // deprecated
            splitter.getString("table"),
            "", "", // deprecated
            splitter.getString("period"),
            new Date(TimeUnit.SECONDS.toMillis(splitter.getLong("start_timestamp_seconds"))),
            new Date(TimeUnit.SECONDS.toMillis(splitter.getLong("end_timestamp_seconds"))),
            splitter.getInt("query_time_millis"),
            0, 0, 0, // deprecated
            splitter.getInt("rows_read"),
            splitter.getInt("rows_ignored"),
            splitter.getIntArray("invalid_rows_ignored_per_id"),
            splitter.getStringArray("metric_ids"),
            EMPTY_STRING_ARRAY, // deprecated
            splitter.getIntArray("send_time_millis_per_id"),
            splitter.getStringArray("storage_per_id"),
            splitter.getIntArray("metrics_sent_per_id"),
            splitter.getInt("total_metrics_count_in_group"),
            splitter.getString("query_weight")
        );
    }

    private void parseTabSeparated(String line, ParserContext context) throws Exception {
        String[] splits = line.split("\t");
        int i = 0;
        Date date = dateFormat.parse(splits[i++]);
        String type = splits[i++];
        String table = splits[i++];
        String[] metricIds = splits[i++].split(",");
        String[] storagePerId = splits[i++].split(",");
        String period = splits[i++];
        Date startDate = ParseUtils.parseDateInSeconds(splits[i++]);
        Date endDate = ParseUtils.parseDateInSeconds(splits[i++]);
        Integer queryTimeMillis = Integer.valueOf(splits[i++]);
        Integer[] sendTimeMillisPerId = ParseUtils.parseIntArray(splits[i++]);
        Integer[] metricsSendPerId = ParseUtils.parseIntArray(splits[i++]);
        Integer metricsIgnored = Integer.valueOf(splits[i++]);
        Integer rowsRead = Integer.valueOf(splits[i++]);
        Integer rowsIgnored = Integer.valueOf(splits[i++]);

        QueryWeight queryWeight;
        switch (period) {
            case "QUARTER":
            case "MONTH":
            case "WEEK":
            case "DAY":
                queryWeight = QueryWeight.HEAVY;
                break;
            case "HOUR":
                queryWeight = QueryWeight.MEDIUM;
                break;
            default:
                if ("market.trace".equals(table)) {
                    queryWeight = QueryWeight.MEDIUM;
                } else {
                    queryWeight = QueryWeight.LIGHT;
                }
        }

        context.write(
            date, context.getHost(), type, table, "", "", period, startDate, endDate,
            queryTimeMillis, 0, 0, metricsIgnored, rowsRead, rowsIgnored, new Integer[]{},
            metricIds, EMPTY_STRING_ARRAY, sendTimeMillisPerId, storagePerId, metricsSendPerId,
            metricIds.length, queryWeight
        );
    }

    enum Storage {
        GRAPHITE, SOLOMON, STATFACE
    }

    enum QueryWeight {
        LIGHT, MEDIUM, HEAVY
    }
}
