package ru.yandex.market.logshatter.parser.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.LinkedHashMultiset;
import com.google.common.collect.Multiset;
import ru.yandex.common.util.date.DateUtil;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.parser.trace.Environment;
import ru.yandex.market.request.trace.TskvRecordBuilder;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.LongSummaryStatistics;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 28/04/15
 */
public class LogshatterPerformanceLog implements LogParser {
    private static final DateUtil.ThreadLocalDateFormat dateFormat = new DateUtil.ThreadLocalDateFormat(
        "yyyy-MM-dd'T'HH:mm:ss.SSSXXX"
    );

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("config", ColumnType.String),
        new Column("parser", ColumnType.String),
        new Column("table", ColumnType.String),
        new Column("log_batch_count", ColumnType.UInt32),
        new Column("line_count", ColumnType.UInt32),
        new Column("parse_error_count", ColumnType.UInt32),
        new Column("output_count", ColumnType.UInt32),
        new Column("bytes_count", ColumnType.UInt64),
        new Column("first_row_date", ColumnType.DateTime),
        new Column("last_row_date", ColumnType.DateTime),
        new Column("complete_date", ColumnType.DateTime),
        new Column("read_time_ms", ColumnType.UInt32),
        new Column("parse_time_ms", ColumnType.UInt32),
        new Column("output_time_ms", ColumnType.Int32),
        new Column("full_output_time_ms", ColumnType.Int32),
        new Column("timestamp_seconds_to_line_count_keys", ColumnType.ArrayUInt32, "[toUInt32(first_row_date)]"),
        new Column("timestamp_seconds_to_line_count_values", ColumnType.ArrayUInt32, "[output_count]"),
        new Column("schema", ColumnType.String),
        new Column("status", EnumColumnType.enum8(OutputStatus.class), "'SUCCESS'"),
        new Column("save_try_number", ColumnType.UInt32, "1"),
        new Column("source_names", ColumnType.ArrayString),
        new Column("environment", EnumColumnType.enum8(Environment.class), "'UNKNOWN'")
    );

    private final EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        TskvSplitter tskvSplitter = new TskvSplitter(line);
        Date firstRowDate = dateFormat.parse(tskvSplitter.getString("first_row_date"));
        Integer outputCount = tskvSplitter.getInt("output_count");
        context.write(
            dateFormat.parse(tskvSplitter.getString("date")),
            context.getHost(),
            tskvSplitter.getString("config"),
            tskvSplitter.getString("parser"),
            tskvSplitter.getString("table"),
            tskvSplitter.getOptionalInt("log_batch_count", 0),
            tskvSplitter.getInt("line_count"),
            tskvSplitter.getInt("parse_error_count"),
            outputCount,
            tskvSplitter.getLong("bytes_count"),
            firstRowDate,
            dateFormat.parse(tskvSplitter.getString("last_row_date")),
            dateFormat.parse(tskvSplitter.getString("date")),
            tskvSplitter.getInt("read_time_ms"),
            tskvSplitter.getInt("parse_time_ms"),
            tskvSplitter.getInt("output_time_ms"),
            tskvSplitter.getInt("full_output_time_ms"),
            tskvSplitter.getOptionalLongArray(
                "timestamp_seconds_to_line_count_keys",
                new Long[] {TimeUnit.MILLISECONDS.toSeconds(firstRowDate.getTime())}
            ),
            tskvSplitter.getOptionalIntArray(
                "timestamp_seconds_to_line_count_values", new Integer[] {outputCount}
            ),
            tskvSplitter.getString("schema"),
            tskvSplitter.getOptionalEnum("status", OutputStatus.SUCCESS),
            tskvSplitter.getOptionalInt("save_try_number", 1),
            tskvSplitter.getStringArray("source_names"),
            environmentMapper.getEnvironment(context)
        );
    }

    @VisibleForTesting
    public static String format(Date date, OutputStatus status, LogShatterConfig logShatterConfig, List<LogBatch> batches,
                                int outputTimeMillis, int fullOutputTimeMillis, int saveTryNumber) {
        return format(
            date, status, logShatterConfig, new BatchesStats(batches), outputTimeMillis, fullOutputTimeMillis,
            saveTryNumber,
            batches.stream()
                .map(LogBatch::getSourceName)
                .distinct()
                .collect(Collectors.joining(","))
        );
    }

    static String format(Date date, OutputStatus status, LogShatterConfig logShatterConfig, BatchesStats batchesStats,
                         int outputTimeMillis, int fullOutputTimeMillis, int saveTryNumber, String sourceNames) {
        TskvRecordBuilder tskvStringBuilder = new TskvRecordBuilder();
        tskvStringBuilder.add("date", dateFormat.format(date));
        tskvStringBuilder.add("status", status);
        tskvStringBuilder.add("config", logShatterConfig.getShortConfigName());
        tskvStringBuilder.add("parser", logShatterConfig.getShortParserName());
        tskvStringBuilder.add("table", logShatterConfig.getTableName());
        tskvStringBuilder.add("log_batch_count", batchesStats.logBatchCount);
        tskvStringBuilder.add("line_count", batchesStats.lineCount);
        tskvStringBuilder.add("parse_error_count", batchesStats.parseErrorCount);
        tskvStringBuilder.add("output_count", batchesStats.outputCount);
        tskvStringBuilder.add("bytes_count", batchesStats.bytesCount);
        tskvStringBuilder.add("first_row_date", dateFormat.format(new Date(batchesStats.firstRowTimeMillis)));
        tskvStringBuilder.add("last_row_date", dateFormat.format(new Date(batchesStats.lastRowTimeMillis)));
        tskvStringBuilder.add("read_time_ms", batchesStats.readTimeMillis);
        tskvStringBuilder.add("parse_time_ms", batchesStats.parseTimeMillis);
        tskvStringBuilder.add("output_time_ms", outputTimeMillis);
        tskvStringBuilder.add("full_output_time_ms", fullOutputTimeMillis);
        tskvStringBuilder.add("schema", logShatterConfig.getSchema());
        tskvStringBuilder.add("save_try_number", saveTryNumber);
        tskvStringBuilder.add("source_names", sourceNames);

        StringBuilder datesBuilder = new StringBuilder();
        StringBuilder lineCountsBuilder = new StringBuilder();
        for (Long timestampSeconds : batchesStats.lineCountsPerTimestampSeconds.elementSet()) {
            datesBuilder.append(timestampSeconds).append(",");
            lineCountsBuilder.append(batchesStats.lineCountsPerTimestampSeconds.count(timestampSeconds)).append(",");
        }

        if (datesBuilder.length() > 0) {
            datesBuilder.deleteCharAt(datesBuilder.length() - 1);
        }

        if (lineCountsBuilder.length() > 0) {
            lineCountsBuilder.deleteCharAt(lineCountsBuilder.length() - 1);
        }

        tskvStringBuilder.add("timestamp_seconds_to_line_count_keys", datesBuilder.toString());
        tskvStringBuilder.add("timestamp_seconds_to_line_count_values", lineCountsBuilder.toString());

        return tskvStringBuilder.build();
    }

    private static Date truncateDate(Date date, long truncateToMillis) {
        return new Date(date.getTime() - (date.getTime() % truncateToMillis));
    }

    public enum OutputStatus {
        SUCCESS, FAILURE
    }

    @VisibleForTesting
    static class BatchesStats {
        private static final long ONE_DAY_MILLIS = TimeUnit.DAYS.toMillis(1);
        private static final long SIX_HOURS_MILLIS = TimeUnit.HOURS.toMillis(6);
        private static final long ONE_HOUR_MILLIS = TimeUnit.HOURS.toMillis(1);
        private static final long FIFTEEN_MINUTES_MILLIS = TimeUnit.MINUTES.toMillis(15);
        private static final long TEN_MINUTES_MILLIS = TimeUnit.MINUTES.toMillis(10);
        private static final long FIVE_MINUTES_MILLIS = TimeUnit.MINUTES.toMillis(5);
        private static final long TWO_MINUTES_MILLIS = TimeUnit.MINUTES.toMillis(2);
        private static final long ONE_MINUTE_MILLIS = TimeUnit.MINUTES.toMillis(1);
        private static final long FIFTEEN_SECONDS_MILLIS = TimeUnit.SECONDS.toMillis(15);
        private static final long FIVE_SECONDS_MILLIS = TimeUnit.SECONDS.toMillis(5);

        int logBatchCount = 0;
        int lineCount = 0;
        int parseErrorCount = 0;
        int outputCount = 0;
        long bytesCount = 0;

        long firstRowTimeMillis;
        long lastRowTimeMillis;

        long readTimeMillis = 0;
        long parseTimeMillis = 0;
        Multiset<Long> lineCountsPerTimestampSeconds = LinkedHashMultiset.create();

        BatchesStats() {
        }

        BatchesStats(Collection<LogBatch> batches) {
            this.logBatchCount = batches.size();
            for (LogBatch batch : batches) {
                lineCount += batch.getLinesCount();
                parseErrorCount += batch.getParseErrors();
                outputCount += batch.getOutputSize();
                bytesCount += batch.getBatchSizeBytes();
                readTimeMillis += batch.getReadDuration().toMillis();
                parseTimeMillis += batch.getParseDuration().toMillis();

                long currentTimeMillis = System.currentTimeMillis();
                for (Date date : batch.getParsedDates()) {
                    long truncatedTimestampMillis = truncateDate(currentTimeMillis, date).getTime();
                    lineCountsPerTimestampSeconds.add(TimeUnit.MILLISECONDS.toSeconds(truncatedTimestampMillis));
                }
            }
            LongSummaryStatistics summaryStatistics = batches.stream()
                .map(LogBatch::getParsedDates)
                .flatMap(Collection::stream)
                .collect(Collectors.summarizingLong(Date::getTime));
            firstRowTimeMillis = Math.min(summaryStatistics.getMin(), 0);
            lastRowTimeMillis = Math.max(summaryStatistics.getMax(), 0);
        }

        @VisibleForTesting
        static Date truncateDate(long currentTimeMillis, Date date) {
            long diffWithCurrentTimeMillis = currentTimeMillis - date.getTime();

            // больше суток — суточная
            if (diffWithCurrentTimeMillis > ONE_DAY_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, ONE_DAY_MILLIS);
            }
            // старше 6 часов — часовая
            else if (diffWithCurrentTimeMillis > SIX_HOURS_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, ONE_HOUR_MILLIS);
            }
            // старше часа — 10-минутная
            else if (diffWithCurrentTimeMillis > ONE_HOUR_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, TEN_MINUTES_MILLIS);
            }
            // старше 15 минут — минутная
            else if (diffWithCurrentTimeMillis > FIFTEEN_MINUTES_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, ONE_MINUTE_MILLIS);
            }
            // старше 5 минут — 15-секундная
            else if (diffWithCurrentTimeMillis > FIVE_MINUTES_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, FIFTEEN_SECONDS_MILLIS);
            }
            // старше 2 минут — пятисекундная
            else if (diffWithCurrentTimeMillis > TWO_MINUTES_MILLIS) {
                return LogshatterPerformanceLog.truncateDate(date, FIVE_SECONDS_MILLIS);
            }

            return date;
        }
    }
}
