package ru.yandex.market.logshatter.parser.ir;

import com.google.common.base.Splitter;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/04/16
 */
public class ScTransitionsParser implements LogParser {

    private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

    private static final int DEFAULT_INT_VALUE = -2;

    public static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList("host", "offer_id"),

        new Column("host", ColumnType.String),
        new Column("offer_id", ColumnType.String),
        new Column("good_id", ColumnType.String),
        new Column("ware_md5", ColumnType.String),
        new Column("category_id", ColumnType.Int32),
        new Column("vendor_id", ColumnType.Int32),
        new Column("shop_id", ColumnType.Int32),
        new Column("feed_id", ColumnType.Int32),
        new Column("shop_offer_id", ColumnType.String),
        new Column("status", ColumnType.String),

        new Column("old_category_id", ColumnType.Int32),
        new Column("new_category_id", ColumnType.Int32),
        new Column("old_mapped_id", ColumnType.Int32),
        new Column("new_mapped_id", ColumnType.Int32),
        new Column("old_classification_type", ColumnType.String),
        new Column("new_classification_type", ColumnType.String),
        new Column("old_classifier_category_id", ColumnType.Int32),
        new Column("new_classifier_category_id", ColumnType.Int32),
        new Column("old_matched_category_id", ColumnType.Int32),
        new Column("new_matched_category_id", ColumnType.Int32),

        new Column("old_vendor_id", ColumnType.Int32),
        new Column("new_vendor_id", ColumnType.Int32),

        new Column("old_matched_type", ColumnType.String),
        new Column("new_matched_type", ColumnType.String),
        new Column("old_matched_id", ColumnType.Int32),
        new Column("new_matched_id", ColumnType.Int32),

        new Column("old_local_vendor_id", ColumnType.Int32),
        new Column("new_local_vendor_id", ColumnType.Int32),
        new Column("old_model_id", ColumnType.Int32),
        new Column("new_model_id", ColumnType.Int32),
        new Column("old_modification_id", ColumnType.Int32),
        new Column("new_modification_id", ColumnType.Int32),

        new Column("old_light_match_type", ColumnType.String),
        new Column("new_light_match_type", ColumnType.String),
        new Column("old_light_model_id", ColumnType.Int32),
        new Column("new_light_model_id", ColumnType.Int32),
        new Column("old_light_modification_id", ColumnType.Int32),
        new Column("new_light_modification_id", ColumnType.Int32),

        new Column("old_clutch_type", ColumnType.String),
        new Column("new_clutch_type", ColumnType.String),
        new Column("old_cluster_id", ColumnType.Int64),
        new Column("new_cluster_id", ColumnType.Int64),
        new Column("old_long_cluster_id", ColumnType.Int64),
        new Column("new_long_cluster_id", ColumnType.Int64)
    );


    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        Map<String, String> values = Splitter.on('\t').withKeyValueSeparator('=').split(line);

        //Обязательные поля
        Date date = dateFormat.parse(values.get("date"));
        String offerId = values.get("offer_id");
        String goodId = values.get("classifier_good_id");
        String wareMd5 = values.get("ware_md5");
        Integer categoryId = Integer.valueOf(values.get("category_id"));
        Integer vendorId = Integer.valueOf(values.get("vendor_id"));
        Integer shopId = Integer.valueOf(values.get("shop_id"));
        Integer feedId = Integer.valueOf(values.get("feed_id"));
        String shopOfferId = String.valueOf(values.get("shop_offer_id"));
        String status = values.get("status");

        context.write(
            date, context.getHost(),
            offerId, goodId, wareMd5, categoryId, vendorId, shopId, feedId, shopOfferId, status,

            getOptionalInt(values, "old_category_id"), getOptionalInt(values, "new_category_id"),
            getOptionalInt(values, "old_mapped_id"), getOptionalInt(values, "new_mapped_id"),
            getOptionalString(values, "old_classification_type"), getOptionalString(values, "new_classification_type"),
            getOptionalInt(values, "old_classifier_category_id"), getOptionalInt(values, "new_classifier_category_id"),
            getOptionalInt(values, "old_matched_category_id"), getOptionalInt(values, "new_matched_category_id"),

            getOptionalInt(values, "old_vendor_id"), getOptionalInt(values, "new_vendor_id"),

            getOptionalString(values, "old_matched_type"), getOptionalString(values, "new_matched_type"),
            getOptionalInt(values, "old_matched_id"), getOptionalInt(values, "new_matched_id"),

            getOptionalInt(values, "old_local_vendor_id"), getOptionalInt(values, "new_local_vendor_id"),
            getOptionalInt(values, "old_model_id"), getOptionalInt(values, "new_model_id"),
            getOptionalInt(values, "old_modification_id"), getOptionalInt(values, "new_modification_id"),


            getOptionalString(values, "old_light_match_type"), getOptionalString(values, "new_light_match_type"),
            getOptionalInt(values, "old_light_model_id"), getOptionalInt(values, "new_light_model_id"),
            getOptionalInt(values, "old_light_modification_id"), getOptionalInt(values, "new_light_modification_id"),

            getOptionalString(values, "old_clutch_type"), getOptionalString(values, "new_clutch_type"),
            getOptionalLong(values, "old_cluster_id"), getOptionalLong(values, "new_cluster_id"),
            getOptionalLong(values, "old_long_cluster_id"), getOptionalLong(values, "new_long_cluster_id")
        );
    }

    private Long getOptionalLong(Map<String, String> values, String filed) {
        String stringValue = values.get(filed);
        if (stringValue == null || stringValue.isEmpty()) {
            return Long.valueOf(DEFAULT_INT_VALUE);
        }
        return Long.valueOf(stringValue);
    }

    private Integer getOptionalInt(Map<String, String> values, String filed) {
        String stringValue = values.get(filed);
        if (stringValue == null || stringValue.isEmpty()) {
            return DEFAULT_INT_VALUE;
        }
        return Integer.valueOf(stringValue);
    }

    private String getOptionalString(Map<String, String> values, String filed) {
        return values.getOrDefault(filed, "");
    }
}
