package ru.yandex.market.logshatter.parser.java;

import com.google.common.annotations.VisibleForTesting;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.trace.Environment;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 02.08.17
 */
public class JavaParallelGcLogParser implements LogParser {
    private static final long MICROS_IN_SECOND = TimeUnit.SECONDS.toMicros(1);
    private static final long MILLIS_IN_SECOND = TimeUnit.SECONDS.toMillis(1);

    private final EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    public static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("module", ColumnType.String),
        new Column("gc_type", EnumColumnType.enum8(GcType.class)),
        new Column("cause", EnumColumnType.enum8(Cause.class)),
        new Column("young_generation_usage_before_kb", ColumnType.Int32, "-1"),
        new Column("young_generation_usage_after_kb", ColumnType.Int32, "-1"),
        new Column("young_generation_size_kb", ColumnType.Int32, "-1"),
        new Column("old_generation_usage_before_kb", ColumnType.Int32, "-1"),
        new Column("old_generation_usage_after_kb", ColumnType.Int32, "-1"),
        new Column("old_generation_size_kb", ColumnType.Int32, "-1"),
        new Column("total_heap_usage_before_kb", ColumnType.Int32, "-1"),
        new Column("total_heap_usage_after_kb", ColumnType.Int32, "-1"),
        new Column("total_heap_size_kb", ColumnType.Int32, "-1"),
        new Column("gc_duration_microsec", ColumnType.UInt32),
        new Column("user_time_millis", ColumnType.UInt32),
        new Column("system_time_millis", ColumnType.UInt32),
        new Column("real_time_millis", ColumnType.UInt32),
        new Column("environment", EnumColumnType.enum8(Environment.class))
    );

    private static final Pattern pattern = Pattern.compile(
        "(?<date>.*?): [\\d\\\\.,]+: \\[(?<gcType>.*?) \\((?<cause>.*?)\\) " +
            "\\[PSYoungGen: (?<youngGenerationUsageBeforeKb>\\d+)K->(?<youngGenerationUsageAfterKb>\\d+)K\\((?<youngGenerationSizeKb>\\d+)K\\)]" +
            "( \\[ParOldGen: (?<oldGenerationUsageBeforeKb>\\d+)K->(?<oldGenerationUsageAfterKb>\\d+)K\\((?<oldGenerationSizeKb>\\d+)K\\)])? " +
            "(?<totalHeapUsageBeforeKb>\\d+)K->(?<totalHeapUsageAfterKb>\\d+)K\\((?<totalHeapSizeKb>\\d+)K\\)" +
            "(, \\[Metaspace: \\d+K->\\d+K\\(\\d+K\\)])?, " +
            "(?<gcDurationSeconds>[\\d\\\\.,]+) secs] \\[Times: user=(?<userTimeSeconds>[\\d\\\\.,]+) sys=(?<systemTimeSeconds>[\\d\\\\.,]+), real=(?<realTimeSeconds>[\\d\\\\.,]+) secs]"
    );

    private static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSZ", Locale.US);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        Matcher matcher = pattern.matcher(line.trim());
        if (!matcher.matches()) {
            return;
        }

        Date date = dateFormat.parse(matcher.group("date"));

        context.write(
            date,
            context.getHost(),
            getModuleName(context.getFile().getFileName().toString()),
            parseGcType(matcher.group("gcType")),
            parseCause(matcher.group("cause")),
            ParseUtils.parseInt(matcher.group("youngGenerationUsageBeforeKb"), -1),
            ParseUtils.parseInt(matcher.group("youngGenerationUsageAfterKb"), -1),
            ParseUtils.parseInt(matcher.group("youngGenerationSizeKb"), -1),
            ParseUtils.parseInt(matcher.group("oldGenerationUsageBeforeKb"), -1),
            ParseUtils.parseInt(matcher.group("oldGenerationUsageAfterKb"), -1),
            ParseUtils.parseInt(matcher.group("oldGenerationSizeKb"), -1),
            ParseUtils.parseInt(matcher.group("totalHeapUsageBeforeKb"), -1),
            ParseUtils.parseInt(matcher.group("totalHeapUsageAfterKb"), -1),
            ParseUtils.parseInt(matcher.group("totalHeapSizeKb"), -1),
            parseSecondsAsMicroseconds(matcher.group("gcDurationSeconds")),
            parseSecondsAsMilliseconds(matcher.group("userTimeSeconds")),
            parseSecondsAsMilliseconds(matcher.group("systemTimeSeconds")),
            parseSecondsAsMilliseconds(matcher.group("realTimeSeconds")),
            environmentMapper.getEnvironment(context)
        );
    }

    @VisibleForTesting
    static String getModuleName(String fileName) {
        int index = fileName.indexOf(".log.gc");

        if (index == -1) {
            index = fileName.indexOf(".gc.log.");
        }

        if (index == -1) {
            throw new IllegalArgumentException("Unable to determine module for file name " + fileName);
        }

        return fileName.substring(0, index);
    }

    private int parseSecondsAsMicroseconds(String gcDurationSeconds) {
        return (int) (Float.parseFloat(gcDurationSeconds.replace(",", ".")) * MICROS_IN_SECOND);
    }

    private int parseSecondsAsMilliseconds(String gcDurationSeconds) {
        return (int) (Float.parseFloat(gcDurationSeconds.replace(",", ".")) * MILLIS_IN_SECOND);
    }

    private GcType parseGcType(String value) {
        switch (value) {
            case "GC":
                return GcType.GC;
            case "Full GC":
                return GcType.FULL_GC;
            default:
                throw new IllegalArgumentException("Unknown GC type value: " + value);
        }
    }

    private Cause parseCause(String value) {
        switch (value) {
            case "System.gc()":
                return Cause.SYSTEM_GC;
            case "Allocation Failure":
                return Cause.ALLOCATION_FAILURE;
            case "Ergonomics":
                return Cause.ERGONOMICS;
            case "GCLocker Initiated GC":
                return Cause.GCLOCKER_INITIATED_GC;
            case "Metadata GC Threshold":
                return Cause.METADATA_GC_THRESHOLD;
            case "Heap Inspection Initiated GC":
                return Cause.HEAP_INSPECTION;
            case "Heap Dump Initiated GC":
                return Cause.HEAP_DUMP;
            case "Last ditch collection":
                return Cause.LAST_DITCH_COLLECTION;
            default:
                throw new IllegalArgumentException("Unknown GC cause value: " + value);
        }
    }

    enum GcType {
        GC, FULL_GC
    }

    /**
     * http://netflix.github.io/spectator/en/latest/ext/jvm-gc-causes/
     * http://hg.openjdk.java.net/jdk8u/jdk8u/hotspot/file/tip/src/share/vm/gc_interface/gcCause.cpp
     */
    enum Cause {
        ALLOCATION_FAILURE,
        GCLOCKER_INITIATED_GC,
        METADATA_GC_THRESHOLD,
        SYSTEM_GC,
        HEAP_INSPECTION,
        HEAP_DUMP,
        LAST_DITCH_COLLECTION,
        ERGONOMICS
    }
}
