package ru.yandex.market.logshatter.parser.marketout;

import com.google.common.base.Splitter;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;

import java.util.*;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 28/07/15
 */
public class ClickDaemonRedirLogParser implements LogParser {

    private static final Set<String> REQUIRED_FIELDS = new HashSet<>(Arrays.asList("show_time", "HTTP_REFERER"));

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("click_host", ColumnType.String),
        new Column("show_time", ColumnType.DateTime),
        new Column("show_host", ColumnType.String),
        new Column("pp", ColumnType.Int32),
        new Column("pof", ColumnType.Int32),
        new Column("clid", ColumnType.Int32),
        new Column("at", ColumnType.Int8),
        new Column("dtype", ColumnType.String),
        new Column("cpa", ColumnType.Int8),
        new Column("referer", ColumnType.String),
        new Column("shop_id", ColumnType.Int32),
        new Column("price", ColumnType.Float64),
        new Column("fee", ColumnType.Float64),
        new Column("yandexuid", ColumnType.String),
        new Column("cp", ColumnType.Int32),
        new Column("url_type", ColumnType.Int32)
    );

    @Override
    public void parse(String line, ParserContext context) throws Exception {

        List<String> params = Splitter.on("@@").splitToList(line);

        Map<String, String> kvParams = new HashMap<>();
        for (String param : params) {
            int splitIndex = param.indexOf('=');
            if (splitIndex > 0) {
                kvParams.put(param.substring(0, splitIndex), param.substring(splitIndex + 1));
            }
        }

        for (String requiredField : REQUIRED_FIELDS) {
            if (!kvParams.containsKey(requiredField)) {
                return;
            }
        }

        Date clickDate = ParseUtils.parseDateInSeconds(params.get((params.size() - 3)));
        Date showDate = ParseUtils.parseDateInSeconds(kvParams.get("show_time"));
        String showHost = ParseUtils.deNull(kvParams.get("host"));
        String referer = kvParams.get("HTTP_REFERER");

        Integer pp = ParseUtils.parseInt(kvParams.get("pp"), -1);
        Integer pof = ParseUtils.parseInt(kvParams.get("pof"), -1);
        Integer clid = ParseUtils.parseInt(kvParams.get("clid"), -1);
        Integer at = ParseUtils.parseInt(kvParams.get("at"), -1);
        Integer cpa = ParseUtils.parseInt(kvParams.get("cpa"), -1);

        String dtype = ParseUtils.deNull(kvParams.get("dtype"));

        Integer shopId = ParseUtils.parseInt(kvParams.get("shop_id"), -1);
        Double price = ParseUtils.parseDouble(kvParams.get("price"), Double.NaN);
        Double fee = ParseUtils.parseDouble(kvParams.get("fee"), Double.NaN);
        String yandexUid = ParseUtils.deNull(kvParams.get("yandexuid"));

        Integer cp = ParseUtils.parseUnsignedInt(kvParams.get("cp"), 0);
        Integer urlType = ParseUtils.parseInt(kvParams.get("url_type"), -1);

        context.write(
            clickDate, context.getHost(), showDate, showHost,
            pp, pof, clid, at, dtype, cpa, referer, shopId, price, fee, yandexUid, cp, urlType
        );

    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
