package ru.yandex.market.logshatter.parser.marketout;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.nio.file.Path;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;

/**
 * https://st.yandex-team.ru/MARKETHEALTH-421.
 *
 * @author kukabara
 */
public class Kpi1LogParser implements LogParser {
    protected final DateFormat dateFormat = new SimpleDateFormat("[yyyy-MM-dd HH:mm:ss Z]");
    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList("source"),
        new Column("host", ColumnType.String),
        new Column("weight", ColumnType.Int64),
        new Column("value", ColumnType.Int64),
        new Column("source", ColumnType.String)
    );

    private enum Settings {
        //date generation feed_id weight start finish status value
        FEED_AGE_LOG_PARSER("feedage", 0, 3, 7),

        //date generation feed_id weight start finish status value
        DIFF_FEED_AGE_LOG_PARSER("diff_feedage", 0, 3, 7),

        //date feed_id weight value
        SCAGE_LOG_PARSER("scage", 0, 2, 3),

        //date o_in_c suc_ap_of ap_err_of rec_mod_of weight this_t_ap_of_cnt_d this_t_ap_of_cnt value type place
        BID_ID_LOG_PARSER("bid_id", 0, 5, 8),

        //date feed_id weight value
        QINDEX_FEEDAGE_LOG_PARSER("qindex", 0, 2, 3),

        //date feed_id weight value
        QINDEX_FEEDAGE_NO_OR_LOG_PARSER("qindex_no_or", 0, 2, 3);

        private String source;
        private int indexDate;
        private int indexWeigth;
        private int indexValue;

        Settings(String source, int indexDate, int indexWeigth, int indexValue) {
            this.source = source;
            this.indexDate = indexDate;
            this.indexWeigth = indexWeigth;
            this.indexValue = indexValue;
        }
    }

    @Override
    @SuppressWarnings("checkstyle:magicnumber")
    public void parse(String line, ParserContext context) throws Exception {
        Settings settings;
        Path path = context.getFile();
        String fileName = path.getName(path.getNameCount() - 1).toString();
        switch (fileName) {
            case "feedage.log":
                settings = Settings.FEED_AGE_LOG_PARSER;
                break;
            case "diff_feedage.log":
                settings = Settings.DIFF_FEED_AGE_LOG_PARSER;
                break;
            case "scage.log":
                settings = Settings.SCAGE_LOG_PARSER;
                break;
            case "mbi-bidding-kpi.log":
                // Брать записи только с type == "FEED_OFFER_ID". Нас интересуют только ставки, поставленные через API
                String splits[] = line.split("\t");
                if (!splits[9].equals("FEED_OFFER_ID")) {
                    return;
                }
                settings = Settings.BID_ID_LOG_PARSER;
                break;
            case "qindex_feedage.log":
                settings = Settings.QINDEX_FEEDAGE_LOG_PARSER;
                break;
            case "qindex_feedage_no_or.log":
                settings = Settings.QINDEX_FEEDAGE_NO_OR_LOG_PARSER;
                break;
            default:
                return;
        }

        parse(line, context, settings);
    }

    public void parse(String line, ParserContext context, Settings settings) throws ParseException {
        String[] splits = line.split("\t");

        Date date = dateFormat.parse(splits[settings.indexDate]);
        long weight = Long.parseLong(splits[settings.indexWeigth]);
        long value = Long.parseLong(splits[settings.indexValue]);

        context.write(
            date, context.getHost(), weight, value, settings.source
        );
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
