package ru.yandex.market.logshatter.parser.marketout;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TableDescription;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 25/03/15
 * Описание формата:
 * https://wiki.yandex-team.ru/Market/Development/Logs/external#access-logimarketnogoreportayandex-market-report
 */
public class ReportLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("full_elapsed", ColumnType.Int32),
        new Column("search_elapsed", ColumnType.Int32),
        new Column("base_search_elapsed", ColumnType.Int32),
        new Column("meta_search_elapsed", ColumnType.Int32),
        new Column("total_rendered", ColumnType.UInt32),
        new Column("total_documents_processed", ColumnType.UInt32),
        new Column("total_documents_accepted", ColumnType.UInt32),
        new Column("req_wiz_time", ColumnType.Int32),
        new Column("error", ColumnType.Int8),
        new Column("redirect", ColumnType.Int8),
        new Column("wizards", ColumnType.UInt16),
        new Column("place", ColumnType.String),
        new Column("pp", ColumnType.UInt32),
        new Column("rids", ColumnType.UInt32),
        new Column("pof", ColumnType.UInt32),
        new Column("reqid", ColumnType.String),
        new Column("wizards_array", ColumnType.ArrayString),
        new Column("market_request_id", ColumnType.String),
        new Column("fetch_elapsed", ColumnType.Int32),
        new Column("fetch_docs", ColumnType.Int32),
        new Column("external_requests_time", ColumnType.Int32),
        new Column("recommendations_type", ColumnType.String),
        new Column("service_request", ColumnType.Int8),
        new Column("partial_answer", ColumnType.Int8),
        new Column("estimated_max_memory_usage", ColumnType.UInt64),
        new Column("test_buckets", ColumnType.ArrayUInt32),
        new Column("env", ColumnType.String),
        new Column("cluster", ColumnType.UInt8),
        new Column("clid", ColumnType.UInt32),
        new Column("bot", ColumnType.String),
        new Column("cpu_time_us", ColumnType.UInt64)
    );

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        ReportTskvLogEntry entry = new ReportTskvLogEntry(line);
        context.write(
            entry.getDate(), context.getHost(),
            entry.getFullElapsedMillis(), entry.getSearchElapsed(),
            entry.getBaseSearchElapsed(), entry.getMetaSearchElapsed(),
            entry.getTotalRendered(), entry.getTotalDocumentsProcessed(), entry.getTotalDocumentsAccepted(),
            entry.getReqWizTimeMillis(), entry.isError(), entry.isRedirect(), entry.getWizards().length,
            entry.getPlace(), entry.getPP(), entry.getRids(), entry.getPP(), //TODO похоже на багу (pp в pof), стоит поправить
            entry.getStatBlockId(), entry.getWizards(), entry.getMarketRequestId(),
            entry.getFetchElapsed(), entry.getFetchDocs(), entry.getExternalRequestsTime(),
            entry.getRecommendationsType(), isServiceRequest(entry), entry.isPartialAnswer(), entry.getEstimatedMaxMemoryUsage(),
            entry.getTestBuckets(), entry.getEnv(), entry.getCluster(), entry.getClid(),
            BotDetector.detect(entry.getUserAgent()).name(), entry.getCpuTimeUs()
        );
    }

    public static boolean isServiceRequest(ReportTskvLogEntry entry) throws ParserException {
        final String place = entry.getPlace();
        // Служебные плейсы.
        if (place.equals("consistency_check") || place.equals("report_status")) {
            return true;
        }
        // Подзапросы.
        if (!ParseUtils.extractStringParam(entry.getUrl(), "rep-outgoing").isEmpty()) {
            return true;
        }
        if (!ParseUtils.extractStringParam(entry.getUrl(), "subrequest").isEmpty()) {
            return true;
        }
        // Запросы от прогревалки.
        if (!ParseUtils.extractStringParam(entry.getUrl(), "local-sources-only").isEmpty()) {
            return true;
        }
        if (!ParseUtils.extractStringParam(entry.getUrl(), "mini-tank").isEmpty()) {
            return true;
        }
        return false;
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

}
