package ru.yandex.market.logshatter.parser.marketout;

import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TskvSplitter;

import java.util.Date;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 15/09/16
 */
public class ReportTskvLogEntry {

    private final TskvSplitter values;
    private final String url;

    private String[] wizards;


    public ReportTskvLogEntry(String line) throws ParserException {
        values = new TskvSplitter(line);
        url = values.getString("url");
    }

    private void ensureWizards() throws ParserException {
        if (wizards == null) {
            wizards = values.getStringArray("wizards");
        }
    }

    public String getMarketRequestId() throws ParserException {
        return values.getString("x_market_req_id");
    }

    public boolean isValid() throws ParserException {
        return !values.getString("error_info").startsWith("Monitoring status");
    }

    public Date getDate() throws ParserException {
        return new Date(getRequestStartTimeMillis() + getFullElapsedMillis());
    }

    public String getUrl() throws ParserException {
        return url;
    }

    public Long getRequestStartTimeMillis() throws ParserException {
        return values.getLong("unixtime_ms");
    }

    public Integer getFullElapsedMillis() throws ParserException {
        return values.getInt("full_elapsed");
    }

    public Integer getSearchElapsed() throws ParserException {
        return values.getInt("search_elapsed");
    }

    public Integer getBaseSearchElapsed() throws ParserException {
        return values.getInt("base_search_elapsed");
    }

    public Integer getMetaSearchElapsed() throws ParserException {
        return values.getInt("meta_search_elapsed");
    }

    public Integer getTotalRendered() throws ParserException {
        return values.getInt("total_rendered");
    }

    public Integer getTotalDocumentsProcessed() throws ParserException {
        return values.getInt("total_documents_processed");
    }

    public Integer getTotalDocumentsAccepted() throws ParserException {
        return values.getInt("total_documents_accepted");
    }

    public Integer getReqWizTimeMillis() throws ParserException {
        return values.getInt("req_wiz_time");
    }

    public Boolean isError() throws ParserException {
        return values.getBoolean("error_info");
    }

    public Boolean isPartialAnswer() throws ParserException {
        return values.getBoolean("partial_answer");
    }

    public Boolean isRedirect() throws ParserException {
        return values.getBoolean("redirect_info");
    }

    public String getPlace() throws ParserException {
        return ParseUtils.extractStringParam(url, "place");
    }

    public Integer getPP() throws ParserException {
        return ParseUtils.extractUnsignedIntegerParam(url, "pp");
    }

    public Integer getClid() throws ParserException {
        String clidStr = ParseUtils.extractStringParam(url, "clid");
        int dashPos = clidStr.indexOf('-');
        if (dashPos != -1) {
            // @see MARKETOUT-22408
            // The parameter may contain string of "<clid>-<vid>" format; we need only clid and must omit vid
            clidStr = clidStr.substring(0, dashPos);
        }
        return ParseUtils.parseUnsignedInt(clidStr, 0);
    }

    public Integer getRids() throws ParserException {
        return ParseUtils.extractUnsignedIntegerParam(url, "rids");
    }

    public String[] getWizards() throws ParserException {
        ensureWizards();
        return wizards;
    }

    public String getRemoteAddress() throws ParserException {
        return values.getString("remote_addr");
    }

    public String getError() throws ParserException {
        return values.getString("error_info");
    }

    public Integer[] getTestIds() throws ParserException {
        return values.getIntArray("test_ids");
    }

    public String getStatBlockId() throws ParserException {
        return values.getString("reqid");
    }

    public Integer getFetchElapsed() throws ParserException {
        return values.getInt("fetch_time");
    }

    public Integer getFetchDocs() throws ParserException {
        return values.getInt("snippets_fetched");
    }

    public Integer getExternalRequestsTime() throws ParserException {
        return values.getInt("external_requests_time");
    }

    public String getRecommendationsType() throws ParserException {
        return ParseUtils.extractStringParam(url, "type");
    }

    public Long getEstimatedMaxMemoryUsage() throws ParserException {
        return values.getLong("estimated_max_memory_usage");
    }

    public Integer[] getTestBuckets() throws ParserException {
        return ParseUtils.parseTestBuckets(ParseUtils.extractDecodedStringParam(url, "test-buckets"));
    }

    public String getEnv() throws ParserException {
        return values.getString("env");
    }

    public Integer getCluster() throws ParserException {
        return values.getInt("cluster");
    }

    public String getUserAgent() throws ParserException {
        return values.getString("user_agent");
    }

    public Long getCpuTimeUs() throws ParserException {
        return values.getLong("cpu_time_us");
    }
}
