package ru.yandex.market.logshatter.parser.marketout;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.JsonLineParser;
import ru.yandex.market.logshatter.parser.KeyValueExtractor;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.util.Date;

/**
 * @author Anton Chugunov <a href="mailto:achugunov@yandex-team.ru"></a>
 */
public class RtyStatsLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("ts", ColumnType.UInt32),
        new Column("hub_ts", ColumnType.UInt32),
        new Column("download_ts", ColumnType.UInt32),
        new Column("feed_id", ColumnType.String),
        new Column("offer_id", ColumnType.String),
        new Column("msku_id", ColumnType.String),
        new Column("price", ColumnType.Float64),
        new Column("old_price", ColumnType.Float64),
        new Column("currency", ColumnType.UInt32),
        new Column("type", ColumnType.String),
        new Column("env", ColumnType.String),
        new Column("report_sub_role", ColumnType.String),
        new Column("report_status", ColumnType.UInt16),
        new Column("report_color", ColumnType.String)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        KeyValueExtractor extractor = createKeyValueExtractor(line);

        Date timestamp = new Date(extractor.getLong("timestamp") * 1000);

        Integer logTime = extractor.getInt("timestamp");
        Integer downloadTime = extractor.getInt("download_ts");
        Integer hubTime = extractor.getInt("hub_ts");
        String feedId = extractor.getString("feed_id");
        String offerId = extractor.getString("offer_id");
        String mskuId = extractor.getOptionalString("msku_id", "");
        Double price = extractor.getOptionalDouble("price", Double.NaN);
        Double oldPrice = extractor.getOptionalDouble("old_price", Double.NaN);
        Integer currency = extractor.getOptionalInt("currency", 0);
        String type = extractor.getString("type");
        String env = extractor.getString("env");
        String reportSubRole = extractor.getString("report_sub_role");
        Integer reportStatus = extractor.getInt("report_status");
        String reportColor = getReportColor(reportSubRole);

        // Skip data from shadow and bk clusters
        if (reportSubRole.endsWith("shadow") || reportSubRole.equals("bk")) {
            return;
        }

        context.write(
            timestamp, context.getHost(),
            logTime, hubTime, downloadTime, feedId, offerId, mskuId, price, oldPrice, currency, type,
            env, reportSubRole, reportStatus, reportColor
        );
    }

    protected KeyValueExtractor createKeyValueExtractor(String line) {
        return new JsonLineParser(line);
    }

    private String getReportColor(String reportSubRole) {
        String result = "white";
        if (reportSubRole.startsWith("blue")) {
            result = "blue";
        } else if (reportSubRole.startsWith("red")) {
            result = "red";
        }
        return result;
    }
}
