package ru.yandex.market.logshatter.parser.mbi;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.ComplicatedLineParser;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

public class ApiQueryProfilerLogParser implements LogParser {
    private static final String DATE_FORMAT = "yyyy-MM-dd HH:mm:ss,S";

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("url", ColumnType.String),
        new Column("requestId", ColumnType.String),
        new Column("type", ColumnType.String),
        new Column("queryType", ColumnType.String),
        new Column("duration", ColumnType.UInt32)
    );

    private ThreadLocal<SimpleDateFormat> formatter = ThreadLocal.withInitial(ApiQueryProfilerLogParser::dateFormat);

    public static SimpleDateFormat dateFormat() {
        return new SimpleDateFormat(DATE_FORMAT);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
    // Valid:
    // [2016-04-29 15:37:01,632] DEBUG [328046478@RequestThread-0] QuerySum /get-cpa-shops#5 JDBC : 705 ms.
    // [2016-04-29 15:37:01,632] DEBUG [328046478@RequestThread-0] Request /get-cpa-shops#5 took 731 ms.
    //
    // Not valid:
    // [2016-04-29 17:21:32,173] INFO  [MemoryStateReporter-Thread] Free memory 31MB (34% of total), total memory 90MB, (78% of max), max memory 114MB

    @Override
    @SuppressWarnings("checkstyle:magicnumber")
    public void parse(String line, ParserContext context) throws Exception {
        String[] parts = line.split("\\s");
        if (parts.length < 5) {
            return;
        }

        String type = parts[4];
        String requestId;
        String url;
        String queryType;
        long duration;
        switch (type) {
            case "QuerySum": {
                requestId = parts[5];
                url = requestId.split("#")[0];
                queryType = parts[6];
                duration = Long.parseLong(parts[8]);
            }
            break;
            case "Request": {
                requestId = parts[5];
                url = requestId.split("#")[0];
                queryType = "";
                duration = Long.parseLong(parts[7]);
            }
            break;
            default:
                return;
        }
        Date date = parseDate(parts[0] + " " + parts[1]);
        context.write(date, context.getHost(), url, requestId, type, queryType, duration);


    }

    private Date parseDate(String dateString) throws ParseException {
        ComplicatedLineParser complicatedLineParser = new ComplicatedLineParser();
        complicatedLineParser.setLine(dateString);
        return formatter.get().parse(complicatedLineParser.getStringSurroundedBy('[', ']', true));
    }
}
