package ru.yandex.market.logshatter.parser.mbi;

import java.util.Arrays;
import java.util.List;
import java.util.OptionalInt;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.MoreObjects;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.nginx.NginxTskvLogEntry;
import ru.yandex.market.logshatter.parser.trace.Environment;
import ru.yandex.market.logshatter.url.Level;
import ru.yandex.market.logshatter.url.Levels;
import ru.yandex.market.logshatter.url.Page;
import ru.yandex.market.logshatter.url.PageMatcher;

/**
 * Log parser для access-tskv логов mbi-partner-stat.
 *
 * @author Kirill Batalin (batalin@yandex-team.ru)
 */
public class MbiPartnerStatLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList("vhost", "http_code"),
        new Column("host", ColumnType.String),
        new Column("vhost", ColumnType.String),
        new Column("url", ColumnType.String),
        new Column("http_method", ColumnType.String),
        new Column("page_id", ColumnType.String),
        new Column("http_code", ColumnType.UInt16),
        new Column("resptime_ms", ColumnType.Int32),
        new Column("client_ip", ColumnType.String),
        new Column("test_id", ColumnType.ArrayInt32),
        new Column("yandex_uid", ColumnType.String),
        new Column("yandex_login", ColumnType.String),
        new Column("client_reqtime_ms", ColumnType.Int32),
        new Column("bytes_sent", ColumnType.Int32),
        new Column("req_id", ColumnType.String),
        new Column("tvm", ColumnType.String),
        new Column("uid", ColumnType.Int64),
        new Column("euid", ColumnType.Int64),
        new Column("campaign_id", ColumnType.Int64),
        new Column("environment", EnumColumnType.enum8(Environment.class), "'UNKNOWN'")
    );

    private EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(final String line, final ParserContext context) throws Exception {
        final PageMatcher pageMatcher = context.getPageMatcher();
        final NginxTskvLogEntry entry = new NginxTskvLogEntry(line, pageMatcher);
        final Page page = getPage(pageMatcher, entry);

        final long campaignId = getCampaignId(page, entry.getUrl());

        context.write(
            entry.getDateTime(), context.getHost(), entry.getVHost(), entry.getUrl(),
            entry.getHttpMethod(), page.getId(), entry.getHttpCode(), entry.getRespTimeMillis(),
            entry.getClientIp(), entry.getTestIds(), entry.getYandexUid(),
            entry.getYandexLogin(), entry.getClientReqTimeMillis(),
            entry.getBytesSent(), entry.getReqId(), entry.getTvm().name(),
            entry.getLongUrlParam("_user_id", -1L),
            entry.getLongUrlParam("euid", -1L), campaignId,
            environmentMapper.getEnvironment(context)
        );
    }

    private long getCampaignId(final Page page, final String url) {
        final List<Level> levels = Levels.parse(page.getPattern());
        final OptionalInt campaignIdIndex = IntStream.range(0, levels.size())
            .filter(i -> levels.get(i).getType() == Level.Type.PATH)
            .filter(i -> "<campaignid>".equals(levels.get(i).getName()))
            .findFirst();

        if (!campaignIdIndex.isPresent()) {
            return -1L;
        }

        // Если передан id кампании, то берем его
        // В компоненте есть тест, который чекает все урлы
        final List<Level> urlLevels = Levels.parse(url);
        final String campaignId = urlLevels.get(campaignIdIndex.getAsInt()).getName();
        return ParseUtils.parseLong(campaignId, -1L);
    }

    private Page getPage(final PageMatcher pageMatcher, final NginxTskvLogEntry entry) throws ParserException {
        final Page page = pageMatcher.matchUrl(entry.getVHost(), entry.getHttpMethod(), entry.getUrl());
        return MoreObjects.firstNonNull(page, Page.EMPTY);
    }
}
