package ru.yandex.market.logshatter.parser.mbi;

import org.apache.commons.lang3.BooleanUtils;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.trace.RequestId;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

/**
 * Парсер для логов партнёрского API
 * Пишутся - https://github.yandex-team.ru/market-java/mbi/blob/master/partner-api/src/java/ru/yandex/market/api/partner/log/impl/PartnerApiAccessLog.java
 *
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 24/02/15
 */
public class PartnerApiAccessLogParser extends BasePartnerApiLogParser {

    private static final Pattern SYSTEM_URL_PATTERN = Pattern.compile("/ping|/biddingMonitoring.*");

    private static final int DATE = 0;
    private static final int REQUEST_METHOD = 1;
    public static final int URL = 3;
    private static final int HTTP_CODE = 6;
    private static final int API_RESOURCE = 7;
    private static final int GROUP_ID = 8;
    private static final int UID = 9;
    private static final int CAMPAIGN_ID = 10;
    private static final int RESP_LENGTH = 11;
    private static final int RESP_TIME = 12;
    private static final int TIMESTAMP = 13;
    private static final int CLIENT_ID = 14;
    private static final int API_COST = 15;
    private static final int TRACE_ID = 16;
    private static final int PARTNER_TYPE = 17;

    private final DateFormat dateFormat = new SimpleDateFormat("[EEE MMM dd HH:mm:ss yyyy]", Locale.US);
    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
            new Column("host", ColumnType.String),
            new Column("api_resource", ColumnType.String),
            new Column("http_code", ColumnType.UInt16),
            new Column("resptime_ms", ColumnType.Int32),
            new Column("group_id", ColumnType.UInt32),
            new Column("uid", ColumnType.UInt64),
            new Column("campaign_id", ColumnType.UInt64),
            new Column("url", ColumnType.String),
            new Column("client_id", ColumnType.String),
            new Column("api_cost", ColumnType.UInt32),
            new Column("request_method", ColumnType.String),
            new Column("is_system_url", ColumnType.UInt8),
            new Column("trace_timestamp_millis", ColumnType.UInt64),
            new Column("trace_hash", ColumnType.String),
            new Column("trace_seq", ColumnType.ArrayUInt32),
            new Column("partner_type", ColumnType.String)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    /*
        [Thu Feb 26 18:37:33 2015]	GET	HTTP/1.1	/v2/campaigns.json?oauth_token=ushdbf8erfg938fgq483734gfeks3fsd&oauth_client_id=b9cb08eef6194d95b3dc1da3e90bfdea&oauth_login=
        77.88.2.30	"Apache-HttpClient/4.3.1 (java 1.5)"	200	/campaigns	1	288915578	-	263	97

        [Tue Nov 28 12:42:20 2017]      GET     HTTP/1.0        /campaigns.json 2a02:6b8:c0e:11d:0:1407:bde6:c421       "-"     200     /campaigns      1       318947777       -       692
413     1511862140      -       1       1511862140339/e94af87b35d40c45194442a5467132a2

        [Wed Jul 10 15:59:25 2019]      GET     HTTP/1.0        //api.partner.market.yandex.ru/v2/campaigns/21475888/settings.json      2a02:6b8:b010:5092:225:90ff:fe95:b382   "-"     200     /campaigns/settings
   25      642970563       21475888        677     112     1562763565      -       1       1562763565463/2abd4defbeac770aa79db6614b7b4951	SHOP
    */
    @Override
    public void parse(String line, ParserContext context) throws Exception {
        String[] splits = line.split("\t");
        if (!has(splits, RESP_LENGTH)) {
            return;
        }
        Date date = getDate(splits);
        String requestMethod = get(splits, REQUEST_METHOD);
        String url = get(splits, URL);
        Integer httpCode = getInt(splits, HTTP_CODE);
        String apiResource = get(splits, API_RESOURCE);
        Integer respTimeMillis = getInt(splits, RESP_TIME);

        Integer groupId = getInt(splits, GROUP_ID);
        Long uid = getLong(splits, UID);
        String clientId = get(splits, CLIENT_ID, "");
        Long campaignId = getLong(splits, CAMPAIGN_ID);
        if (campaignId < 0) {
            campaignId = 0L;
        }
        Integer apiCost = getInt(splits, API_COST);
        boolean isSystemUrl = url != null && SYSTEM_URL_PATTERN.matcher(url).matches();


        String traceId = get(splits, TRACE_ID, "0/-");
        RequestId requestId = RequestId.parse(traceId);
        String partnerType = get(splits, PARTNER_TYPE, "");

        context.write(
                date, context.getHost(), apiResource, httpCode, respTimeMillis,
                groupId, uid, campaignId, url, clientId, apiCost, requestMethod, BooleanUtils.toInteger(isSystemUrl),
                requestId.getIdMillis(), requestId.getIdHash(), requestId.getIdSeq(), partnerType
        );
    }

    private Date getDate(String[] splits) throws ParseException {
        Date date;
        //Если есть таймстамп, то используем его MBI-13294
        if (has(splits, TIMESTAMP)) {
            Long timestamp = getLong(splits, TIMESTAMP);
            date = new Date(TimeUnit.SECONDS.toMillis(timestamp));
        } else {
            String dateStr = get(splits, DATE);
            date = dateFormat.parse(dateStr);
        }
        return date;
    }
}
