package ru.yandex.market.logshatter.parser.mbo;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

/**
 * @author amaslak
 */
public class MboJettyLogParser implements LogParser {

    static final String DATE_PATTERN = "yyyy-MM-dd'T'HH:mm:ssZ";

    private final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    private static final String FILE_NAME_SUFFIX = "-jetty-health.log";

    private static final String HOST = "host";
    private static final String COMPONENT = "component";

    private static final String REQUESTS_RECEIVED = "requests_received";
    private static final String REQUESTS_PROCESSED = "requests_processed";
    private static final String REQUESTS_ACTIVE = "requests_active";
    private static final String REQUESTS_ACTIVE_MAX = "requests_active_max";
    private static final String REQUESTS_TIME_MIN = "requests_time_min";
    private static final String REQUESTS_TIME_AVG = "requests_time_avg";
    private static final String REQUESTS_TIME_MAX = "requests_time_max";
    private static final String RESPONSES_1XX = "responses_1xx";
    private static final String RESPONSES_2XX = "responses_2xx";
    private static final String RESPONSES_3XX = "responses_3xx";
    private static final String RESPONSES_4XX = "responses_4xx";
    private static final String RESPONSES_5XX = "responses_5xx";

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column(HOST, ColumnType.String),
        new Column(COMPONENT, ColumnType.String),
        new Column(REQUESTS_RECEIVED, ColumnType.Int32),
        new Column(REQUESTS_PROCESSED, ColumnType.Int32),
        new Column(REQUESTS_ACTIVE, ColumnType.Int32),
        new Column(REQUESTS_ACTIVE_MAX, ColumnType.Int32),
        new Column(REQUESTS_TIME_MIN, ColumnType.Int32),
        new Column(REQUESTS_TIME_AVG, ColumnType.Int32),
        new Column(REQUESTS_TIME_MAX, ColumnType.Int32),
        new Column(RESPONSES_1XX, ColumnType.Int32),
        new Column(RESPONSES_2XX, ColumnType.Int32),
        new Column(RESPONSES_3XX, ColumnType.Int32),
        new Column(RESPONSES_4XX, ColumnType.Int32),
        new Column(RESPONSES_5XX, ColumnType.Int32)
    );

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        String filename = context.getFile().getFileName().toString();
        if (!filename.endsWith(FILE_NAME_SUFFIX)) {
            return;
        }
        String component = filename.substring(0, filename.length() - FILE_NAME_SUFFIX.length());

        String[] tokens = line.split(" ");
        if (tokens.length < 13) {
            return;
        }

        String[] dateKeyValue = tokens[0].split("=");

        if (dateKeyValue.length < 2) {
            return;
        }

        Date date = dateFormat.parse(dateKeyValue[1]);

        Map<String, Integer> metrics = new HashMap<>();
        for (int i = 1; i < tokens.length; i++) {
            String token = tokens[i];
            String[] keyValue = token.split("=");
            metrics.put(keyValue[0], Integer.valueOf(keyValue[1]));
        }

        context.write(
            date,
            context.getHost(),
            component,
            metrics.getOrDefault(REQUESTS_RECEIVED, 0),
            metrics.getOrDefault(REQUESTS_PROCESSED, 0),
            metrics.getOrDefault(REQUESTS_ACTIVE, 0),
            metrics.getOrDefault(REQUESTS_ACTIVE_MAX, 0),
            metrics.getOrDefault(REQUESTS_TIME_MIN, 0),
            metrics.getOrDefault(REQUESTS_TIME_AVG, 0),
            metrics.getOrDefault(REQUESTS_TIME_MAX, 0),
            metrics.getOrDefault(RESPONSES_1XX, 0),
            metrics.getOrDefault(RESPONSES_2XX, 0),
            metrics.getOrDefault(RESPONSES_3XX, 0),
            metrics.getOrDefault(RESPONSES_4XX, 0),
            metrics.getOrDefault(RESPONSES_5XX, 0)
        );
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

}
