package ru.yandex.market.logshatter.parser.nginx;

import com.google.common.base.Splitter;
import org.apache.commons.lang3.StringUtils;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.url.Page;
import ru.yandex.market.logshatter.url.PageMatcher;

import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/09/16
 */
public class NginxTskvLogEntry {
    private static final String[] EMPTY_ARRAY_OF_STRINGS = new String[0];

    private static final Set<String> STATIC_EXTENSIONS = new HashSet<>(Arrays.asList(
        "jpeg", "jpg", "gif", "png", "svg", "html", "htm", "txt", "js", "css", "ico", "swf", "woff",
        "fft", "dtd", "image", "eot"
    ));

    protected final TskvSplitter values;
    private final PageMatcher pageMatcher;
    private Page page;
    private Map<String, String> cookies;
    private String url;

    // SortedMap чтобы getInvalidCookieNames и getInvalidCookieValues обходили мапу в одном и том же порядке
    // Нельзя сделать список пар или два списка потому что геттеры кук теоретически могут быть вызваны несколько раз
    private SortedMap<String, String> invalidCookies;


    /*
    tskv   	tskv_format=access-log-cs-vs-tools     	timestamp=2016-09-13T14:19:09  	timezone=+0300 	status=200     	protocol=HTTP/1.1      	method=POST    	request=/gate/route/history/decorate?sk=u2efd539feb869d618cbf20cbd84c1813      	referer=https://market.pepelac1ft.yandex.ru/search?deliveryincluded=0&onstock=0&fesh=248279    	cookies=yandexuid=8678819561469532601; fuid01=5797bc9830b6d578.bKPifSZGLv02H6CAD_ld6sPTclnljcdDKjVGAQS-dEC2haJeRkOhIO-oZ7xznl8_lMDj3mNTNcNrylb6ez47BXMoBvaX86T658mtpjObvjfh7BJqvncnIn8XETQq65o1; mda=0; deliveryincluded=1; ps_gch=8493562674468410368; yandex_gid=213; _ym_uid=1472641861155013662; L=Wlp5RQZoeHZhCw1VbVl1blNjX29TalBZMkA+PzY4FwEjEVwDHwIr.1472641775.12619.342.5cd1b5cbf79bb234b20280d3124fad22; in-stock=1; zm=m-white_bender.flex.webp.css-https%3Awww_15WBbDQd8sMKhiSq6GAGF1xBNHk%3Al%7Cm-_freddie70.css-https%3A%2F%2Fyastatic.net%2Fcovers%2Ffreddie70%2F3%2Fresources%2F_freddie70.css%3Ac; yabs-frequency=/4/0G000CG2l5S00000/V35oS1mc8TWiSd0S9a49Bd9m72OYsInoS10cRgKjSd0S9k00/; yp=1784961738.multib.1#1500664610.st_soft_stripe_s.1#1488853882.szm.2_00:1680x1050:1680x978#1477667635.ww.1#1788001775.udn.cDpCeWtob3Z0c2V2LkVnb3I%3D#1475233738.ygu.1; ys=ymrefl.F698E57B1809A8C7; Cookie_check=1; uid=SL7rYVfRjfw9MyyiAwN9Ag==; parent_reqid_seq=a47449b54717e39de7fe413b6b49c21c%2C68c44a99574f49ad3c65b9d6c5705dcb%2C02e4d56efa2b405adeb473eb466a961d%2C36049dba1da77bdd7ea593b03ab0387c%2C161cac0725a8d14ee853cce5f83ee0aa; Session_id=3:1473603820.5.0.1471425080000:4xWOrFKJUvQMBAAAuAYCKg:e.1|316769815.0.302|34163024.426852.2.2:426852|151423.42846.XXXXXXXXXXXXXXXXXXXXXXXXXXX; sessionid2=3:1473603821.5.0.1471425080000:4xWOrFKJUvQMBAAAuAYCKg:e.1|316769815.0.302|34163024.426852.2.2:426852|151423.806024.XXXXXXXXXXXXXXXXXXXXXXXXXXX; yandex_login=Bykhovtsev.Egor; _ym_isad=2; device_id=\"a4bf581b74af5ddfa5761e60ae7959a7e3af6e194\"; HISTORY_AUTH_SESSION=f5b94627; mxp=nika|30934|27704,0,20%3B30934,0,13%3B30824,0,69%3B15093,0,29%3B31240,0,55|market_cpa_category_pess_mode%253Dcpa-and-cpc%253Bmarket_category_relevance_formula%253Dfull_mode_f%253Bmarket_category_redirect_treshold%253D-1; yandexmarket=10,RUR,1,,,,2,0,0,213; _ym_visorc_160656=b      	user_agent=Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2743.82 Safari/537.36 OPR/39.0.2256.48   	vhost=market.pepelac1ft.yandex.ru      	ip=2a02:6b8:0:c33::1:1d1       	x_forwarded_for=-      	x_real_ip=-    	bytes_sent=2111	req_id=f7c08354faf23555fb0fcf3199c26124	req_id_seq=68c44a99574f49ad3c65b9d6c5705dcb,02e4d56efa2b405adeb473eb466a961d,36049dba1da77bdd7ea593b03ab0387c,161cac0725a8d14ee853cce5f83ee0aa,f7c08354faf23555fb0fcf3199c26124	upstream_resp_time=0.048       	req_time=0.053 	scheme=https   	device_type=desktop    	x_sub_req_id=- 	yandexuid=8678819561469532601  	ssl_handshake_time=0.000       	market_buckets=27704,0,20;30934,0,13;30824,0,69;15093,0,29;31240,0,55  	upstream_addr=unix:/var/run/yandex-market-skubi-exp/nika/server.sock   	upstream_header_time=0.047     	upstream_status=200     request_tags=CROSSBORDER,SHOP
     */
    public NginxTskvLogEntry(String line, PageMatcher pageMatcher) {
        this.values = new TskvSplitter(line);
        this.pageMatcher = pageMatcher;
    }

    public OffsetDateTime getDateTime() throws ParserException {
        String date = values.getString("timestamp");
        String timezone = values.getString("timezone");
        LocalDateTime dateTime = LocalDateTime.parse(date);
        return dateTime.atOffset(ZoneOffset.of(timezone));
    }

    public String getVHost() throws ParserException {
        return getStringDeDash("vhost");
    }

    public String getHttpMethod() throws ParserException {
        return getStringDeDash("method");
    }

    public Integer getHttpCode() throws ParserException {
        // компонент может в http code всегда кидать 200, а ошибку передавать в заголовке x-return-code
        return values.getOptionalInt("x_return_code", values.getInt("status"));
    }

    public String getHttpProtocol() throws ParserException {
        return getStringDeDash("protocol");
    }

    public String getScheme() throws ParserException {
        return getStringDeDash("scheme");
    }

    public Tvm getTvm() {
        return Tvm.getTvm(deDash(values.getOptionalString("tvm", "")));
    }

    public int getRespTimeMillis() throws ParserException {
        return parseServersRespTimeMillis(getStringDeDash("upstream_resp_time"));
    }

    public int getHeaderTimeMillis() throws ParserException {
        return parseServersRespTimeMillis(getStringDeDash("upstream_header_time"));
    }

    public long getRequestEndTimeMillis() throws ParserException {
        Double startTimeMs = values.getDouble("msec") * 1000L;
        return startTimeMs.longValue();
    }

    public int getSslHandshakeTimeMillis() throws ParserException {
        return parseServersRespTimeMillis(getStringDeDash("ssl_handshake_time"));
    }

    public int getClientReqTimeMillis() throws ParserException {
        return parseServersRespTimeMillis(getStringDeDash("req_time"));
    }

    public String getService() throws ParserException {
        return getStringDeDash("device_type");
    }

    public String getUserAgent() throws ParserException {
        return getStringDeDash("user_agent");
    }

    public String getYandexUid() throws ParserException {
        return getStringDeDash("yandexuid");
    }

    public int getRequestLength() {
        return values.getOptionalInt("request_length", 0);
    }

    public int getBytesSent() {
        return values.getOptionalInt("bytes_sent", 0);
    }

    public String getClientIp() throws ParserException {
        return getStringDeDash("ip");
    }

    public String getReferer() throws ParserException {
        return getStringDeDash("referer");
    }

    public String[] getRequestTags() {
        String value = deDash(values.getOptionalString("request_tags", ""));
        return value.isEmpty() ? EMPTY_ARRAY_OF_STRINGS : value.split(",");
    }

    public String getUrl() throws ParserException {
        if (url == null) {
            url = getStringDeDash("request");
        }
        return url;
    }

    public String getYandexLogin() throws ParserException {
        ensureCookies();
        return cookies.getOrDefault("yandex_login", "");
    }

    public String getYandexMarketCookie() throws ParserException {
      ensureCookies();
      return cookies.getOrDefault("yandexmarket", "");
    }

    public String getYandexPortalPermanentMetaCookie() throws ParserException {
      ensureCookies();
      return cookies.getOrDefault("yp", "");
    }

    public String getYandexPortalSessionMetaCookie() throws ParserException {
      ensureCookies();
      return cookies.getOrDefault("ys", "");
    }

    public String getReqId() throws ParserException {
        return getStringDeDash("req_id");
    }

    public String getMarkerRequestId() throws ParserException {
        return values.getOptionalString("market_req_id", "");
    }

    public String getPageId() throws ParserException {
        ensurePage();
        return page.getId();
    }

    public String getPageType() throws ParserException {
        ensurePage();
        return page.getType();
    }

    public String getUpstreamAddress() throws ParserException {
        return values.getString("upstream_addr");
    }

    public int getResponseSizeBytes() throws ParserException {
        return values.getInt("bytes_sent");
    }

    public boolean isDynamic() throws ParserException {
        ensurePage();
        if (page != Page.EMPTY) {
            return true;
        }
        String url = getUrl();
        if (url.startsWith("/_/")) {
            return false;
        }

        String withoutQueryAndFragment = ParseUtils.cutQueryStringAndFragment(url);
        int indexAfterDot = withoutQueryAndFragment.lastIndexOf('.') + 1;
        if (indexAfterDot <= 1) {
            return true; // Если нет расширения - динамика
        }
        String extension = withoutQueryAndFragment.substring(indexAfterDot).toLowerCase();
        return !STATIC_EXTENSIONS.contains(extension);
    }

    public Integer[] getTestIds() throws ParserException {
        return ParseUtils.parseTestBuckets(getStringDeDash("market_buckets"));
    }

    public String[] getInvalidCookieNames() {
        if (invalidCookies == null) {
            return EMPTY_ARRAY_OF_STRINGS;
        }
        return invalidCookies.keySet().toArray(EMPTY_ARRAY_OF_STRINGS);
    }

    public String[] getInvalidCookieValues() {
        if (invalidCookies == null) {
            return EMPTY_ARRAY_OF_STRINGS;
        }
        return invalidCookies.values().toArray(EMPTY_ARRAY_OF_STRINGS);
    }

    public int getYandexGid() throws ParserException {
        return getNumberCookie("yandex_gid", Integer::parseInt, -2, -1);
    }

    public long getEffectiveUserId() throws ParserException {
        return getNumberCookie("euid", Long::parseLong, -2L, -1L);
    }

    public String getStringUrlParam(String name) throws ParserException {
        return ParseUtils.extractStringParam(getUrl(), name);
    }

    public Long getLongUrlParam(String name, Long defaultValue) throws ParserException {
        return ParseUtils.parseLong(getStringUrlParam(name), defaultValue);
    }

    private <T extends Number> T getNumberCookie(String name, Function<String, T> parser,
                                                 T emptyValue, T errorValue) throws ParserException {
        ensureCookies();
        String valueAsString = cookies.get(name);
        if (StringUtils.isEmpty(valueAsString)) {
            return emptyValue;
        }
        try {
            return parser.apply(valueAsString);
        } catch (NumberFormatException e) {
            addInvalidCookie(name, valueAsString);
            return errorValue;
        }
    }

    private void addInvalidCookie(String name, String valueAsString) {
        if (invalidCookies == null) {
            invalidCookies = new TreeMap<>();
        }
        invalidCookies.put(name, valueAsString);
    }

    private void ensureCookies() throws ParserException {
        if (cookies == null) {
            cookies = parseCookies(getStringDeDash("cookies"));
        }
    }

    private static Map<String, String> parseCookies(String string) {
        if (string.isEmpty() || string.equals("-")) {
            return Collections.emptyMap();
        }
        Map<String, String> cookies = new HashMap<>();
        for (String kvString : Splitter.on("; ").split(string)) {
            String[] splits = kvString.split("=", 2);
            if (splits.length < 2) {
                continue;
            }
            cookies.put(splits[0], splits[1]);
        }
        return cookies;
    }

    private void ensurePage() throws ParserException {
        if (page == null) {
            String pageId = deDash(values.getOptionalString("page_id", ""));
            String pageType = deDash(values.getOptionalString("page_type", ""));
            if (pageId.isEmpty() && pageType.isEmpty()) {
                page = pageMatcher.matchUrl(getVHost(), getHttpMethod(), getUrl());
                if (page == null) {
                    page = Page.EMPTY;
                }
            } else {
                page = new Page(pageId, pageType);
            }
        }
    }

    private static int parseServersRespTimeMillis(String respTimes) {
        return (int) (parseServersRespTime(respTimes) * 1000);
    }

    private static double parseServersRespTime(String respTimes) {
        respTimes = respTimes.trim();
        if (respTimes.isEmpty() || respTimes.equals("-")) {
            return 0;
        }

        double timeSeconds = 0;

        if (respTimes.indexOf(':') >= 0) {
            String[] splits = respTimes.split(":");
            for (int i = 0; i < splits.length; i++) {
                timeSeconds += parseServersRespTime(splits[i]);
            }
        } else if (respTimes.indexOf(',') >= 0) {
            String[] splits = respTimes.split(",");
            for (int i = 0; i < splits.length; i++) {
                timeSeconds += parseServersRespTime(splits[i]);
            }
        } else {
            timeSeconds = Double.valueOf(respTimes);
        }
        return timeSeconds;
    }


    protected String getStringDeDash(String key) throws ParserException {
        return deDash(values.getString(key));
    }

    private String deDash(String string) {
        return !string.equals("-") ? string : "";
    }

    public enum Tvm {
        UNKNOWN, DISABLED, NO_TICKET, ERROR, OK;

        public static Tvm getTvm(String tvm) {
            if (tvm == null || tvm.isEmpty()) {
                return Tvm.UNKNOWN;
            }
            try {
                return Tvm.valueOf(tvm);
            } catch (Exception e) {
                return Tvm.UNKNOWN;
            }
        }
    }
}
