package ru.yandex.market.logshatter.parser.startrek;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.reader.startrek.StarTrekHistoryEvent;
import ru.yandex.market.logshatter.reader.startrek.StarTrekLine;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * @author Vlad Vinogradov <a href="mailto:vladvin@yandex-team.ru"></a> on 21.12.16
 */
abstract class StarTrekBaseInfo {

    public static final String STATUS_EVENT_KEY = "status";

    private static final Column[] BASE_COLUMNS = new Column[]{
        new Column("queue", ColumnType.String),
        new Column("issue", ColumnType.String),
        new Column("type", ColumnType.String),
        new Column("createdAt", ColumnType.DateTime),
        new Column("updatedAt", ColumnType.DateTime),
        new Column("resolvedAt", ColumnType.DateTime),
        new Column("priority", ColumnType.String),
        new Column("resolution", ColumnType.String),
        new Column("tags", ColumnType.ArrayString),
        new Column("components", ColumnType.ArrayString),
        new Column("weight", ColumnType.Float32),
        new Column("issueWeight", ColumnType.String),
        new Column("statusChange", ColumnType.ArrayString),
        new Column("statusChangeDateTime", ColumnType.ArrayDateTime),
        new Column("assignee", ColumnType.String),
        new Column("author", ColumnType.String),
    };

    private static <T> T[] mergeArrays(T[] fst, T[] snd) {
        final T[] allValues = Arrays.copyOf(fst, fst.length + snd.length);
        System.arraycopy(snd, 0, allValues, fst.length, snd.length);
        return allValues;
    }

    static TableDescription buildTableDescription(Column... extraColumns) {
        final Column[] allColumns = mergeArrays(BASE_COLUMNS, extraColumns);
        return TableDescription.createDefault(Arrays.asList("queue", "issue"), allColumns);
    }

    static Object[] buildLineInfo(StarTrekLine line, Object... extraValues) {

        List<String> statusChange = new ArrayList<>();
        List<Date> statusChangeDate = new ArrayList<>();

        for (StarTrekHistoryEvent historyEvent : line.getHistory()) {
            if (STATUS_EVENT_KEY.equals(historyEvent.getEventKey())) {
                statusChange.add(historyEvent.getEventValue());
                statusChangeDate.add(new Date(historyEvent.getEventMillis()));
            }
        }

        Object[] baseValues = new Object[]{
            line.getQueue(),
            line.getIssue(),
            line.getType(),
            new Date(line.getCreated()),
            new Date(line.getUpdated()),
            new Date(line.getResolved()),
            line.getPriority(),
            line.getOptionalOrDefault(StarTrekLine.OptField.RESOLUTION, String.class, ""),
            line.getTags().toArray(),
            line.getComponents().toArray(),
            line.getOptionalOrDefault(StarTrekLine.OptField.WEIGHT, Double.class, 0.0).floatValue(),
            line.getOptionalOrDefault(StarTrekLine.OptField.ISSUE_WEIGHT, String.class, ""),
            statusChange,
            statusChangeDate,
            line.getAssignee(),
            line.getCreatedBy()
        };
        return mergeArrays(baseValues, extraValues);
    }
}
