package ru.yandex.market.logshatter.parser.startrek;

import com.google.gson.Gson;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.reader.startrek.StarTrekLine;

import java.util.Date;

/**
 * @author Vlad Vinogradov <a href="mailto:vladvin@yandex-team.ru"></a> on 14.12.16
 */
public class StarTrekIssueParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = StarTrekBaseInfo.buildTableDescription(
        new Column("status", ColumnType.String),
        new Column("sreStartTime", ColumnType.DateTime),
        new Column("sreNotificationTime", ColumnType.DateTime),
        new Column("sreEndTime", ColumnType.DateTime),
        new Column("issueWeightPoints", ColumnType.Int32, "0")
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        final StarTrekLine starTrekLine = new Gson().fromJson(line, StarTrekLine.class);

        long sreStartTimeMillis = starTrekLine.getOptionalOrDefault(StarTrekLine.OptField.SRE_BEGIN_TIME, Double.class, 0.0).longValue();
        long sreNotificationTimeMillis = starTrekLine.getOptionalOrDefault(StarTrekLine.OptField.SRE_NOTIFICATION_TIME, Double.class, 0.0).longValue();
        long sreEndTimeMillis = starTrekLine.getOptionalOrDefault(StarTrekLine.OptField.SRE_END_TIME, Double.class, 0.0).longValue();

        int issueWeightPoints = 0;

        String issueWeight = starTrekLine.getOptionalOrDefault(StarTrekLine.OptField.ISSUE_WEIGHT, String.class, null);
        if (issueWeight != null && !issueWeight.isEmpty() && !issueWeight.equals("-")) {
            // Иногда значения вностся с пробелами между разрядами
            issueWeight = issueWeight.replace(" ", "");
            try {
                issueWeightPoints = Integer.parseInt(issueWeight);
            } catch (NumberFormatException e) {
                issueWeightPoints = -1;
            }
        }

        final Object[] fields = StarTrekBaseInfo.buildLineInfo(
            starTrekLine,
            starTrekLine.getStatus(),
            new Date(sreStartTimeMillis),
            new Date(sreNotificationTimeMillis),
            new Date(sreEndTimeMillis),
            issueWeightPoints
        );

        context.write(new Date(starTrekLine.getCreated()), fields);
    }
}
