package ru.yandex.market.logshatter.parser.trace;

import com.google.common.base.MoreObjects;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TskvSplitter;
import ru.yandex.market.logshatter.url.Page;
import ru.yandex.market.logshatter.url.PageMatcher;

import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

public class TraceLogEntry {
    private static final String KV_KEY_PREFIX = "kv.";
    private static final String EVENT_NAME_PREFIX = "event.";
    // ZonedDateTime.parse uses DateTimeFormatter.ISO_ZONED_DATE_TIME
    // that has a bug to not parse correctly either the missing colon or hour-only offset
    // https://bugs.java.com/view_bug.do?bug_id=JDK-8032051
    // Therefore this workaround is needed
    private static final DateTimeFormatter ZONED_DATE_TIME_FORMATTER = new DateTimeFormatterBuilder()
        .append(DateTimeFormatter.ISO_LOCAL_DATE_TIME)
        .optionalStart().appendOffset("+HH:MM", "+00:00").optionalEnd()
        .optionalStart().appendOffset("+HHMM", "+0000").optionalEnd()
        .optionalStart().appendOffset("+HH", "Z").optionalEnd()
        .toFormatter();

    private TskvSplitter values;
    private String module;
    private RequestId requestId;
    private RequestType requestType;
    private ParserContext context;
    private List<String> kvKeys = new ArrayList<>();
    private List<String> kvValues = new ArrayList<>();
    private List<String> events = new ArrayList<>();
    private List<Long> eventTimestamps = new ArrayList<>();
    private long endTimeMs;
    private int durationMs;
    private Page page;
    private Integer[] testIds;

    public TraceLogEntry(String line, ParserContext context) throws ParserException {
        this.context = context;
        values = new TskvSplitter(line);
        module = extractModuleName(context.getFile());
        requestId = RequestId.parse(values.getString("request_id"));
        requestType = RequestType.valueOf(values.getOptionalString("type", RequestType.OUT.name()));
        endTimeMs = ZonedDateTime.parse(values.getString("date"), ZONED_DATE_TIME_FORMATTER).toInstant().toEpochMilli();
        durationMs = extractDurationMs(values);
        page = extractPage(values, context);
        testIds = extractTestIds(values);

        for (Map.Entry<String, String> entry : values.getValues().entrySet()) {
            if (entry.getKey().startsWith(KV_KEY_PREFIX)) {
                kvKeys.add(entry.getKey().substring(KV_KEY_PREFIX.length()));
                kvValues.add(entry.getValue());
            }

            if (entry.getKey().startsWith(EVENT_NAME_PREFIX)) {
                events.add(entry.getKey().substring(EVENT_NAME_PREFIX.length()));
                eventTimestamps.add(Long.parseUnsignedLong(entry.getValue()));
            }
        }
    }

    public RequestId getRequestId() {
        return requestId;
    }

    public TskvSplitter getValues() {
        return values;
    }

    public String getModule() {
        return module;
    }

    public RequestType getRequestType() {
        return requestType;
    }

    public String getSourceModule() {
        return values.getOptionalString("source_module", requestType.equals(RequestType.OUT) ? module : "");
    }

    public String getSourceHost() {
        return values.getOptionalString("source_host", requestType.equals(RequestType.OUT) ? context.getHost() : "");
    }

    public String getTargetModule() {
        return values.getOptionalString("target_module", requestType.equals(RequestType.IN) ? module : "");
    }

    public String getTargetHost() {
        return values.getOptionalString("target_host", requestType.equals(RequestType.IN) ? context.getHost() : "");
    }

    public String getRequestMethod() {
        return values.getOptionalString("request_method", "");
    }

    public Integer getHttpCode() {
        return values.getOptionalInt("http_code", -1);
    }

    public Integer getRetryNum() {
        return values.getOptionalInt("retry_num", 1);
    }

    public String getErrorCode() {
        return values.getOptionalString("error_code", "");
    }

    public String getProtocol() {
        return values.getOptionalString("protocol", "");
    }

    public String getHttpMethod() {
        return values.getOptionalString("http_method", "");
    }

    public String getQueryParams() {
        return values.getOptionalString("query_params", "");
    }

    public String getYandexUid(){
        return values.getOptionalString("yandex_uid", "");
    }

    public String getYandexLogin(){
        return values.getOptionalString("yandex_login", "");
    }

    public Integer getResponseSizeBytes(){
        return values.getOptionalInt("response_size_bytes", -1);
    }

    public List<String> getKvKeys() {
        return kvKeys;
    }

    public List<String> getKvValues() {
        return kvValues;
    }

    public List<String> getEvents() {
        return events;
    }

    public List<Long> getEventTimestamps() {
        return eventTimestamps;
    }

    public long getEndTimeMs() {
        return endTimeMs;
    }

    public int getDurationMs() {
        return durationMs;
    }

    public long getStartTimeMs() {
        return endTimeMs - durationMs;
    }

    public Page getPage() {
        return page;
    }

    public Integer[] getTestIds() {
        return testIds;
    }

    private String extractModuleName(Path file) {
        String fileName = file.getFileName().toString();
        int index = fileName.indexOf("-request-trace.log");

        if (index == -1) {
            index = fileName.indexOf("-trace.log");
        }

        if (index == -1) {
            return "";
        }

        return fileName.substring(0, index);
    }

    private static int extractDurationMs(TskvSplitter values) {
        Integer durationMs = values.getOptionalInt("time_millis", values.getOptionalInt("duration_ms", null));

        if (durationMs == null) {
            throw new IllegalArgumentException("Neither time_millis nor duration_ms is not found");
        }
        return durationMs;
    }

    private static Page extractPage(TskvSplitter values, ParserContext context) {
        String sourceVirtualHost = values.getOptionalString("source_vhost", "");
        String sourceHttpMethod = values.getOptionalString("source_http_method", "");
        String sourceRequestMethod = values.getOptionalString("source_request_method", "");
        String sourcePageId = values.getOptionalString("source_page_id", "");
        String sourcePageType = values.getOptionalString("source_page_type", "");

        if (!sourcePageId.isEmpty() && !sourcePageType.isEmpty()) {
            return new Page(sourcePageId, sourcePageType);
        } else if (!sourceVirtualHost.isEmpty() && !sourceHttpMethod.isEmpty() && !sourceRequestMethod.isEmpty()) {
            PageMatcher pageMatcher = context.getPageMatcher();
            Page matcherPage = pageMatcher.matchUrl(sourceVirtualHost, sourceHttpMethod, sourceRequestMethod);
            return MoreObjects.firstNonNull(matcherPage, Page.EMPTY);
        } else {
            return Page.EMPTY;
        }
    }

    private static Integer[] extractTestIds(TskvSplitter values) {
        String testIds = values.getOptionalString("test_ids", null);
        if (testIds == null) {
            return new Integer[]{};
        }

        return Arrays.stream(testIds.split(",")).map(Integer::parseUnsignedInt).toArray(Integer[]::new);
    }

}
