package ru.yandex.market.logshatter.parser.yamarec;

import com.google.gson.Gson;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;


/**
 * This is log parser for real-time log uploader 'yandex-yamarec-rt-logger'.
 * Used in Market Recommendations.
 *
 * @author Anton Chugunov <a href="mailto:achugunov@yandex-team.ru"></a>
 * @date 29/09/15
 */
public class YamarecRTLoggerLogParser implements LogParser {

    private final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,S Z");
    private final String logDataMarker = "stat:";

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("log_type", ColumnType.String),
        new Column("upload_events", ColumnType.Int64),
        new Column("delay", ColumnType.UInt32),
        new Column("read_events", ColumnType.UInt64),
        new Column("target", ColumnType.String)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    /**
     * Here is example of log line (fields delimited by \t):
     * 2015-10-01 21:58:10,171 +0300    MONRUN   13689  UploadManager   Partition stat: {"last_timestamp": 1443633240, "upload_events": 4, "delay": 6, "log_type": "mrec-log", "partition": "rt3.iva--cs-http--mrec-log:0"}
     * 2015-10-01 21:58:10,171 +0300\tMONRUN\t13689\tUploadManager\tPartition stat...
     */
    @Override
    @SuppressWarnings("checkstyle:magicnumber")
    public void parse(String line, ParserContext context) throws Exception {
        String lineParts[] = line.split("\t");
        if (lineParts[1].equals("MONRUN")) {
            Date timestamp = dateFormat.parse(lineParts[0]);
            int logDataIndex = lineParts[4].indexOf(logDataMarker) + logDataMarker.length();
            Gson gson = new Gson();
            String logData = lineParts[4].substring(logDataIndex);
            LogRecordData recData = gson.fromJson(logData, LogRecordData.class);
            context.write(
                timestamp,
                context.getHost(),
                recData.log_type,
                recData.upload_events,
                recData.delay,
                recData.read_events,
                recData.target);
        }
    }

    private class LogRecordData {
        String log_type;
        String partition;
        long read_events;
        long upload_events;
        long last_timestamp;
        int delay;
        String target;
    }
}
