package ru.yandex.market.logshatter.reader;

import com.google.common.base.Preconditions;
import com.google.common.base.Stopwatch;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.leader.LeaderLatch;
import org.apache.curator.framework.recipes.leader.LeaderLatchListener;
import org.apache.curator.framework.recipes.leader.LeaderSelector;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListenerAdapter;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.curator.framework.state.ConnectionStateListener;
import org.apache.curator.retry.RetryNTimes;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.net.InetAddress;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/07/15
 */
public class ReadSemaphore implements InitializingBean, ConnectionStateListener, Runnable {

    private static final Logger log = LogManager.getLogger();
    private static final String ZK_LOCK_DIR = "/lock/";

    private LogShatterMonitoring monitoring;
    private int queueSizeMb = 512;
    private int maxOpenFiles = 512;
    private String zookeeperQuorum;
    private int zookeeperTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(30);
    private String zookeeperPrefix;

    private int clickhouseTimeoutSeconds;

    private CuratorFramework curatorFramework;
    private String host;

    private QueuesLimits limits = new QueuesLimits();

    private volatile boolean running = true;

    private final MonitoringUnit zkConnectionMonitoringUnit = new MonitoringUnit("ZooKeeper");
    private final MonitoringUnit openFilesMonitoringUnit = new MonitoringUnit("OpenFiles");
    private final MonitoringUnit lastOutputMonitoringUnit = new MonitoringUnit("LastRead");
    private final MonitoringUnit lastReadMonitoringUnit = new MonitoringUnit("LastRead");

    private final ReentrantLock canReadLock = new ReentrantLock();
    private final Condition canReadCondition = canReadLock.newCondition();
    private final Semaphore openFilesSemaphore = new Semaphore(0, true);

    private final AtomicLong lastReadTimeMillis = new AtomicLong(System.currentTimeMillis());
    private final AtomicLong lastOutputTimeMillis = new AtomicLong(System.currentTimeMillis());

    private final AtomicLong globalQueueCurrentSizeBytes = new AtomicLong();

    private final Map<String, QueuesCounter> sourceIdToQueuesCounterMap = new ConcurrentHashMap<>();

    private final Object queuesLock = new Object();
    private List<AtomicLong> queuesCurrentSizeBytes;
    private List<Long> queuesLimitsBytes;

    @Override
    public void afterPropertiesSet() throws Exception {
        Preconditions.checkState(
            TimeUnit.MILLISECONDS.toSeconds(zookeeperTimeoutMillis) >= clickhouseTimeoutSeconds,
            "Zookeeper timeout must be greater or at least equal to Clickhouse timeout. " +
                "It helps to prevent duplicates when logshatter lost connection to ZK, but still writes batches " +
                "to Clickhouse"
        );

        monitoring.getHostCritical().addUnit(openFilesMonitoringUnit);
//        monitoring.addUnit(lastReadMonitoringUnit);
//        monitoring.addUnit(lastOutputMonitoringUnit);
        openFilesSemaphore.release(maxOpenFiles);
        host = InetAddress.getLocalHost().getCanonicalHostName();
        if (zookeeperQuorum != null) {
            curatorFramework = CuratorFrameworkFactory.builder()
                .connectString(zookeeperQuorum)
                .retryPolicy(new RetryNTimes(Integer.MAX_VALUE, zookeeperTimeoutMillis))
                .connectionTimeoutMs(zookeeperTimeoutMillis)
                .sessionTimeoutMs(zookeeperTimeoutMillis)
                .namespace(zookeeperPrefix)
                .build();
            curatorFramework.getConnectionStateListenable().addListener(this);
            curatorFramework.start();
            monitoring.getClusterCritical().addUnit(zkConnectionMonitoringUnit);
        }

        queuesCurrentSizeBytes = limits.getLimits().stream().map(limit -> new AtomicLong())
            .collect(Collectors.toCollection(CopyOnWriteArrayList::new));

        queuesLimitsBytes = limits.getLimits().stream().map(QueuesLimits.QueueLimit::getLimitBytes)
            .collect(Collectors.toList());

        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            running = false;
            curatorFramework.close();
        }));
        new Thread(this, "ReadSemaphore").start();
    }

    @Override
    public void run() {
        if (zookeeperQuorum != null) {
            try {
                if (!curatorFramework.getZookeeperClient().blockUntilConnectedOrTimedOut()) {
                    zkConnectionMonitoringUnit.critical("No connection to zk: " + zookeeperQuorum);
                }
            } catch (InterruptedException ignored) {
            }
        }
        while (!Thread.interrupted()) {
            try {
                TimeUnit.MINUTES.sleep(1);
                reportQueueUsage();
                updateMonitorings();
            } catch (Exception e) {
                log.error("Exception in ReadSemaphore:", e);
            }
        }
    }

    private void updateMonitorings() {
        updateLastOutputMonitoring();
        updateOpenFilesMonitoring();
        updateLastReadMonitoring();
    }

    private void reportQueueUsage() {
        StringBuilder report = new StringBuilder("Queues current size report:\n");
        report.append(
            String.format(
                "queue A size: %d MB\n", LogShatterUtil.bytesToMb(globalQueueCurrentSizeBytes.get())
            )
        );

        for (int i = 0; i < queuesCurrentSizeBytes.size(); ++i) {
            final int index = i;
            String sources = sourceIdToQueuesCounterMap.entrySet()
                .stream().filter(e -> e.getValue().ids.contains(index))
                .map(Map.Entry::getKey)
                .collect(Collectors.joining(","));

            String limitString;
            if (i < limits.getLimits().size()) {
                QueuesLimits.QueueLimit limit = limits.getLimits().get(i);
                limitString = String.format(
                    "limit: %d MB\trule: %s",
                    LogShatterUtil.bytesToMb(limit.getLimitBytes()), limit.getRule().pattern()
                );
            } else {
                limitString = String.format("limit - MB\trule: %s", sources);
            }

            report.append(
                String.format(
                    "queue %d\tsize: %d MB\t%s\tsources: %s\n",
                    i, LogShatterUtil.bytesToMb(queuesCurrentSizeBytes.get(i).get()), limitString, sources
                )
            );
        }

        log.info(report);
    }

    private void updateLastReadMonitoring() {
        long minutesSinceLastRead = TimeUnit.MILLISECONDS.toMinutes(
            System.currentTimeMillis() - lastReadTimeMillis.get()
        );

        if (minutesSinceLastRead > 10) {
            lastReadMonitoringUnit.critical("No new data read for more than 10 minutes");
        } else if (minutesSinceLastRead > 5) {
            lastReadMonitoringUnit.warning("No new data read for more than 5 minutes");
        } else {
            lastReadMonitoringUnit.ok();
        }
    }

    private void updateLastOutputMonitoring() {
        long minutesSinceLastOutput = TimeUnit.MILLISECONDS.toMinutes(
            System.currentTimeMillis() - lastOutputTimeMillis.get()
        );

        if (minutesSinceLastOutput > 10) {
            lastOutputMonitoringUnit.critical("No new data output for more than 10 minutes");
        } else if (minutesSinceLastOutput > 5) {
            lastOutputMonitoringUnit.warning("No new data output for more than 5 minutes");
        } else {
            lastOutputMonitoringUnit.ok();
        }
    }


    @Override
    public void stateChanged(CuratorFramework client, ConnectionState newState) {
        switch (newState) {
            case CONNECTED:
            case RECONNECTED:
                log.info("Got connection to ZK. Status " + newState);
                zkConnectionMonitoringUnit.ok();
                break;
            case LOST:
            case SUSPENDED:
                zkConnectionMonitoringUnit.critical("Lost connection to ZK");
                log.error("Lost connection to ZK. Status " + newState);
                break;
            default:
                throw new IllegalStateException("Unknown connection state:" + newState);
        }
    }

    public LeaderSelector getDistributedLock(String lockName, LeaderSelectorListenerAdapter listener,
                                             ExecutorService executorService) {
        LeaderSelector leaderSelector = new LeaderSelector(
            curatorFramework, ZK_LOCK_DIR + lockName, executorService, listener
        );
        leaderSelector.autoRequeue();
        leaderSelector.setId(host);
        leaderSelector.start();
        return leaderSelector;
    }

    public LeaderLatch getDistributedLock(final String lockName, LeaderLatchListener listener) throws Exception {
        return getDistributedLock(lockName, listener, true);
    }

    public LeaderLatch getDistributedLock(final String lockName, LeaderLatchListener listener,
                                          boolean start) throws Exception {
        LeaderLatch leaderLatch = new LeaderLatch(curatorFramework, ZK_LOCK_DIR + lockName, host);
        if (listener != null) {
            leaderLatch.addListener(listener);
        }
        leaderLatch.addListener(new LeaderLatchListener() {
            @Override
            public void isLeader() {
                log.info("Got leadership for: " + lockName);
            }

            @Override
            public void notLeader() {
                log.info("Lost leadership for: " + lockName);
            }
        });
        if (start) {
            leaderLatch.start();
        }
        return leaderLatch;
    }

    private void updateOpenFilesMonitoring() {
        double freeOpenFilesLimitPercent = openFilesSemaphore.availablePermits() * 100.0 / maxOpenFiles;
        if (freeOpenFilesLimitPercent < 5) {
            openFilesMonitoringUnit.critical("Less than 5% of open files limit available");
        } else if (freeOpenFilesLimitPercent < 20) {
            openFilesMonitoringUnit.warning("Less than 20% of open files limit available");
        } else {
            openFilesMonitoringUnit.ok();
        }
    }

    public void acquireOpenFileSemaphore() throws IOException {
        boolean greenLight = openFilesSemaphore.tryAcquire();
        if (!greenLight) {
            try {
                log.warn("No available open file permits. Waiting one min.");
                greenLight = openFilesSemaphore.tryAcquire(1, TimeUnit.MINUTES);
            } catch (InterruptedException e) {
                throw new IOException("Failed to acquire openFilesSemaphore. Interrupted.", e);
            }
        }
        if (!greenLight) {
            throw new IOException("Failed to acquire openFilesSemaphore. Max open files: " + maxOpenFiles);
        }
    }

    public boolean hasOpenFilesHunger() {
        return openFilesSemaphore.getQueueLength() > 0;
    }

    public void releaseOpenFileSemaphore() {
        openFilesSemaphore.release();
    }

    public int getOpenFilesCount() {
        return maxOpenFiles - openFilesSemaphore.availablePermits();
    }

    /**
     * @return wait time millis
     * @throws InterruptedException
     */
    public long waitForRead() throws InterruptedException {
        Stopwatch stopwatch = Stopwatch.createStarted();
        canReadLock.lock();
        while (!canRead()) {
            canReadCondition.await();
        }
        canReadLock.unlock();
        return stopwatch.stop().elapsed(TimeUnit.MILLISECONDS);
    }

    public void notifyRead() {
        lastReadTimeMillis.set(System.currentTimeMillis());
    }

    public void incrementGlobalQueue(long sizeBytes) {
        if (sizeBytes <= 0) {
            return;
        }

        long currentSizeBytes = globalQueueCurrentSizeBytes.addAndGet(sizeBytes);
        log.debug(
            "Internal queue size increased in " + sizeBytes + "bytes " +
                "(" + LogShatterUtil.bytesToMb(sizeBytes) + "mb). " +
                "Actual size: " + LogShatterUtil.bytesToMb(currentSizeBytes) + "mb."
        );

        notifyRead();
    }

    private void incrementQueues(List<Integer> ids, long sizeBytes) {
        for (Integer id : ids) {
            queuesCurrentSizeBytes.get(id).addAndGet(sizeBytes);
        }
    }

    private void decrementQueues(List<Integer> ids, long sizeBytes) {
        for (Integer id : ids) {
            queuesCurrentSizeBytes.get(id).addAndGet(-sizeBytes);
        }
    }

    private QueuesLimits.QueueLimit getQueueThatReachedLimit(List<Integer> queuesIds) {
        if (queuesIds.isEmpty()) {
            return null;
        }

        if (queuesIds.size() == 1) {
            // in this case queue may have no limit so we should check this too
            int queueId = queuesIds.get(0);
            return queueId < queuesLimitsBytes.size() &&
                queuesCurrentSizeBytes.get(queueId).get() > queuesLimitsBytes.get(queueId) ?
                limits.getLimits().get(queueId) : null;
        }

        for (Integer id : queuesIds) {
            if (queuesCurrentSizeBytes.get(id).get() > queuesLimitsBytes.get(id)) {
                return limits.getLimits().get(id);
            }
        }

        return null;
    }

    public void decrementGlobalQueue(long sizeBytes) {
        if (sizeBytes <= 0) {
            return;
        }

        long currentSizeBytes = globalQueueCurrentSizeBytes.addAndGet(-sizeBytes);
        log.debug(
            "Internal queue size decreased in " + sizeBytes + "bytes " +
                "(" + LogShatterUtil.bytesToMb(sizeBytes) + "mb). " +
                "Actual size: " + LogShatterUtil.bytesToMb(currentSizeBytes) + "mb."
        );

        canReadLock.lock();
        canReadCondition.signalAll();
        canReadLock.unlock();

        lastOutputTimeMillis.set(System.currentTimeMillis());
    }

    public long getQueueSizeBytes() {
        return globalQueueCurrentSizeBytes.get();
    }

    public double getInternalQueueUsagePercent() {
        return Math.min(globalQueueCurrentSizeBytes.get() * 100.0 / mbToBytes(queueSizeMb), 100);
    }

    private boolean canRead() {
        return running && globalQueueCurrentSizeBytes.get() < mbToBytes(queueSizeMb);
    }

    private long mbToBytes(int mb) {
        return mb * 1048576L;
    }

    public void setZookeeperTimeoutSeconds(int zookeeperTimeoutSeconds) {
        this.zookeeperTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(zookeeperTimeoutSeconds);
    }

    @Required
    public void setMonitoring(LogShatterMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setZookeeperQuorum(String zookeeperQuorum) {
        this.zookeeperQuorum = zookeeperQuorum;
    }

    @Required
    public void setZookeeperPrefix(String zookeeperPrefix) {
        this.zookeeperPrefix = zookeeperPrefix;
    }

    public void setQueueSizeMb(int queueSizeMb) {
        this.queueSizeMb = queueSizeMb;
    }

    public void setMaxOpenFiles(int maxOpenFiles) {
        this.maxOpenFiles = maxOpenFiles;
    }

    public void setClickhouseTimeoutSeconds(int clickhouseTimeoutSeconds) {
        this.clickhouseTimeoutSeconds = clickhouseTimeoutSeconds;
    }

    public void setLimits(QueuesLimits limits) {
        this.limits = limits;
    }

    public QueuesCounter getEmptyQueuesCounter() {
        return new QueuesCounter(Collections.emptyList());
    }

    public QueuesCounter getQueuesCounterForSource(String id) {
        QueuesCounter counter = sourceIdToQueuesCounterMap.get(id);
        if (counter != null) {
            return counter;
        }

        List<QueuesLimits.QueueLimit> queueLimits = limits.getLimits().stream()
            .filter(l -> l.check(id))
            .collect(Collectors.toList());

        if (!queueLimits.isEmpty()) {
            sourceIdToQueuesCounterMap.put(
                id, new QueuesCounter(
                    queueLimits.stream().map(QueuesLimits.QueueLimit::getQueueId).collect(Collectors.toList())
                )
            );

            return sourceIdToQueuesCounterMap.get(id);
        }

        synchronized (queuesLock) {
            counter = sourceIdToQueuesCounterMap.get(id);
            if (counter != null) {
                return counter;
            }

            int queueId = queuesCurrentSizeBytes.size();
            queuesCurrentSizeBytes.add(new AtomicLong());

            sourceIdToQueuesCounterMap.put(id, new QueuesCounter(Collections.singletonList(queueId)));
        }

        return sourceIdToQueuesCounterMap.get(id);
    }

    public class QueuesCounter {
        private List<Integer> ids;

        private QueuesCounter(List<Integer> ids) {
            this.ids = ids;
        }

        public void increment(long bytes) {
            incrementQueues(this.ids, bytes);
        }

        public void decrement(long bytes) {
            decrementQueues(this.ids, bytes);
        }

        public QueuesLimits.QueueLimit getQueueThatReachedLimit() {
            return ReadSemaphore.this.getQueueThatReachedLimit(this.ids);
        }
    }
}
