package ru.yandex.market.logshatter.reader;

import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.logging.BatchErrorLogger;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.meta.SourceKey;
import ru.yandex.market.logshatter.parser.LogParser;

import java.nio.file.Path;
import java.util.Deque;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/07/15
 */
public abstract class SourceContext {
    private final Queue<LogBatch> parseQueue = new ConcurrentLinkedQueue<>();
    private final Deque<LogBatch> outputQueue = new ConcurrentLinkedDeque<>();

    private final LogShatterConfig logShatterConfig;
    private final BatchErrorLogger errorLogger;
    private final SourceKey sourceKey;
    private final LogParser logParser;
    private final ReadSemaphore.QueuesCounter queuesCounter;

    protected volatile boolean finished;
    private CompletableFuture<Void> savingFuture;

    public SourceContext(LogShatterConfig logShatterConfig,
                         String origin,
                         String sourceId,
                         BatchErrorLoggerFactory errorLoggerFactory,
                         ReadSemaphore.QueuesCounter queuesCounter) {
        this.logShatterConfig = logShatterConfig;
        this.sourceKey = new SourceKey(origin, sourceId, logShatterConfig.getTableName());
        this.logParser = logShatterConfig.createParser();
        this.errorLogger = errorLoggerFactory.createErrorLogger(this);
        this.queuesCounter = queuesCounter;
    }

    public SourceContext(LogShatterConfig logShatterConfig,
                         SourceKey sourceKey,
                         BatchErrorLoggerFactory errorLoggerFactory,
                         ReadSemaphore.QueuesCounter queuesCounter) {
        this.logShatterConfig = logShatterConfig;
        this.sourceKey = sourceKey;
        this.logParser = logShatterConfig.createParser();
        this.errorLogger = errorLoggerFactory.createErrorLogger(this);
        this.queuesCounter = queuesCounter;
    }

    public String getOrigin() {
        return sourceKey.getOrigin();
    }

    public SourceKey getSourceKey() {
        return sourceKey;
    }

    public abstract boolean isClosed();

    public abstract String getHost();

    public abstract Path getPath();

    public abstract long getDataOffset();

    public abstract void setDataOffset(long dataOffset);

    public abstract long getFileOffset();

    public abstract void setFileOffset(long fileOffset);

    public abstract String getName();

    public abstract int getInstanceId();

    /**
     * Имеет смысл только для источника logbroker
     */
    public String getLogBrokerTopic() {
        return null;
    }

    public LogShatterConfig getLogShatterConfig() {
        return logShatterConfig;
    }

    public Map<String, String> getParams() {
        return logShatterConfig.getParams();
    }

    public Queue<LogBatch> getParseQueue() {
        return parseQueue;
    }

    public Deque<LogBatch> getOutputQueue() {
        return outputQueue;
    }

    public LogParser getLogParser() {
        return logParser;
    }

    public BatchErrorLogger getErrorLogger() {
        return errorLogger;
    }

    /**
     * Означает, что обработка данных из этого SourceContext должна быть прекращена ASAP.
     * parser-worker'ы могут не парсить вычитанные строки, output-worker'ы могут не сохранять распаршенные батчи.
     */
    public boolean isFinished() {
        return finished;
    }

    /**
     * @return Возвращает true, если можно сохранять.
     */
    public synchronized boolean beginSave() {
        if (finished) {
            return false;
        }
        savingFuture = new CompletableFuture<>();
        return true;
    }

    public synchronized void completeSave() {
        if (savingFuture != null) {
            savingFuture.complete(null);
            savingFuture = null;
        }
    }

    public synchronized CompletableFuture<Void> finish() {
        finished = true;
        if (savingFuture != null) {
            return savingFuture;
        } else {
            return CompletableFuture.completedFuture(null);
        }
    }

    public synchronized void resetFinishedState() {
        finished = false;
    }

    public ReadSemaphore.QueuesCounter getQueuesCounter() {
        return queuesCounter;
    }

    public long getCreationTimeOfTheLastBatchInParseQueue() {
        return getCreationTimeOfTheLastBatchIn(parseQueue);
    }

    public long getCreationTimeOfTheLastBatchInOutputQueue() {
        return getCreationTimeOfTheLastBatchIn(outputQueue);
    }

    private static long getCreationTimeOfTheLastBatchIn(Queue<LogBatch> queue) {
        LogBatch logBatch = queue.peek();
        return logBatch == null ? Long.MAX_VALUE : logBatch.getCreationTimeMillis();
    }
}
