package ru.yandex.market.logshatter.reader.file;

import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.reader.LogReader;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.SourceContext;

import java.nio.file.Path;
import java.util.Collections;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 16/02/15
 */
public class FileContext extends SourceContext {

    private final Object fileKey;
    private final String host;
    private volatile LogReader logReader;
    private volatile Path path;
    private volatile long dataOffset = 0;
    private volatile boolean removed = false;
    private volatile boolean reading = false;

    /**
     * Позиция полностью обработанного места в файле
     */
    private volatile long fileOffset = 0;
    /**
     * Позиция прочитанного места в файле
     */
    private volatile long readerFilePosition = 0;
    private volatile long readerDataPosition = 0;
    private volatile long knowLastModified = 0;
    private volatile long knownSizeBytes = 0;

    public FileContext(LogShatterConfig logShatterConfig, String origin, Path path,
                       Object fileKey, String host, BatchErrorLoggerFactory errorLoggerFactory,
                       ReadSemaphore.QueuesCounter queuesCounter) {
        super(logShatterConfig, origin, fileKey.toString(), errorLoggerFactory, queuesCounter);
        this.path = path;
        this.fileKey = fileKey;
        this.host = host;
    }

    public long getReaderFilePosition() {
        return readerFilePosition;
    }

    @Override
    public void setFileOffset(long fileOffset) {
        this.fileOffset = fileOffset;
        if (fileOffset > readerFilePosition) {
            setReaderFilePosition(fileOffset);
        }
    }

    public void setReaderFilePosition(long readerFilePosition) {
        this.readerFilePosition = readerFilePosition;
        if (readerFilePosition > knownSizeBytes) {
            knownSizeBytes = readerFilePosition;
        }
    }

    public void setKnownSizeBytes(long knownSizeBytes) {
        this.knownSizeBytes = knownSizeBytes;
    }

    public long getKnownSizeBytes() {
        return knownSizeBytes;
    }

    public boolean isFileUpdated() {
        return knownSizeBytes > readerFilePosition;
    }

    public void setLogReader(LogReader logReader) {
        this.logReader = logReader;
    }

    public long getKnowLastModified() {
        return knowLastModified;
    }

    public void setKnowLastModified(long knowLastModified) {
        this.knowLastModified = knowLastModified;
    }

    public LogReader getLogReader() {
        return logReader;
    }

    @Override
    public String getHost() {
        return host;
    }

    @Deprecated
    public String getFileName() {
        return path.toString();
    }

    @Override
    public String getName() {
        return path.toString();
    }

    @Override
    public int getInstanceId() {
        return 0;
    }

    @Override
    public boolean isClosed() {
        return removed;
    }

    public void setRemoved(boolean removed) {
        this.removed = removed;
    }

    public void setPath(Path path) {
        this.path = path;
    }

    public Path getPath() {
        return path;
    }

    @Override
    public long getDataOffset() {
        return dataOffset;
    }

    @Override
    public void setDataOffset(long dataOffset) {
        this.dataOffset = dataOffset;
    }

    @Override
    public long getFileOffset() {
        return fileOffset;
    }

    public long getReaderDataPosition() {
        return readerDataPosition;
    }

    public void setReaderDataPosition(long readerDataPosition) {
        this.readerDataPosition = readerDataPosition;
    }

    public Object getFileKey() {
        return fileKey;
    }

    public String getFileId() {
        return fileKey.toString();
    }

    public boolean isReading() {
        return reading;
    }

    public void setReading(boolean reading) {
        this.reading = reading;
    }

    @Override
    public String toString() {
        return "FileContext{" +
            "path=" + path +
            '}';
    }
}
