package ru.yandex.market.logshatter.reader.file;

import com.google.common.base.Stopwatch;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.logshatter.reader.AbstractReaderService;
import ru.yandex.market.logshatter.reader.LogReader;
import ru.yandex.market.logshatter.reader.ReadQueue;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 05/08/15
 */
public class FileReaderService extends AbstractReaderService implements InitializingBean, Runnable {


    private FileWatcherService fileWatcherService;

    private final ConcurrentHashMap<FileContext, Object> openedFiles = new ConcurrentHashMap<>();
    private final ReadQueue readQueue = new ReadQueue();


    private final MonitoringUnit fileWatcherMonitoringUnit = new MonitoringUnit("FileWatcher");
    private final MonitoringUnit readQueueMonitoringUnit = new MonitoringUnit("ReadQueue");

    private int readThreadCount = 4;
    private volatile int readBatchSizeMb = 10;
    private volatile int fileTimeSinceModificationToCloseMinutes = 15;

    private ExecutorService readExecutorService;

    @Override
    public void afterPropertiesSet() throws Exception {
        monitoring.getHostCritical().addUnit(fileWatcherMonitoringUnit);
        monitoring.getHostCritical().addUnit(readQueueMonitoringUnit);

        readExecutorService = createExecutorService(readThreadCount, "file-");
        for (int i = 0; i < readThreadCount; i++) {
            readExecutorService.submit(new ReaderWorker());
        }
        new Thread(this).start();
    }

    @Override
    public void run() {
        while (!Thread.interrupted() && isRunning()) {
            try {
                List<FileContext> fileContexts = fileWatcherService.processAll();
                readQueue.addAll(fileContexts);
                closeUselessFiles();
                fileWatcherMonitoringUnit.ok();
            } catch (Exception e) {
                log.error("Failed to get changes", e);
                fileWatcherMonitoringUnit.critical(e);
            }
            updateReadQueueMonitoring();
            try {
                TimeUnit.SECONDS.sleep(5);
            } catch (InterruptedException ignored) {
            }

        }
        log.info("Finishing");
    }


    private void updateReadQueueMonitoring() {
        long readQueueMb = LogShatterUtil.bytesToMb(readQueue.bytesToRead());
        if (readQueueMb > 0){
            log.info("File read queue {}Mb", readQueueMb);
        }
        if (readQueueMb > 102400) {
            readQueueMonitoringUnit.warning("More then 10Gb in read queue");
        } else {
            readQueueMonitoringUnit.ok();
        }
    }


    private class ReaderWorker implements Runnable {
        @Override
        public void run() {
            while (!Thread.interrupted() && isRunning()) {
                try {
                    read();
                } catch (InterruptedException ignored) {
                }
            }
            log.info("Reader thread finished");

        }

        private void read() throws InterruptedException {
            FileContext fileContext = readQueue.takeLock();
            fileContext.setReading(true);
            try {
                if (!fileContext.isFileUpdated()) {
                    return;
                }
                if (!openLogReader(fileContext)) {
                    return;
                }
                read(fileContext);
            } catch (IOException e) {
                log.error("Failed to read data from file: " + fileContext.getName(), e);
            } finally {
                fileContext.setReading(false);
                maybeCloseLogReader(fileContext);
                readQueue.returnLock(fileContext);
            }
        }

        private void read(FileContext fileContext) throws InterruptedException, IOException {
            while (!Thread.interrupted() && isRunning()) {
                readSemaphore.waitForRead();

                LogBatch logBatch = readLogBatch(fileContext);
                if (logBatch == null) {
                    return;
                }

                readSemaphore.incrementGlobalQueue(logBatch.getBatchSizeBytes());

                fileContext.getParseQueue().add(logBatch);
                addToParseQueue(fileContext);
            }
        }

        private LogBatch readLogBatch(FileContext fileContext) throws IOException {
            Stopwatch stopwatch = Stopwatch.createStarted();

            LogReader logReader = fileContext.getLogReader();
            long startPosition = logReader.getDataPosition();
            long batchSizeBytes = 0;

            ArrayList<String> lines = new ArrayList<>();
            String line;

            long filePosition = -1;
            while (batchSizeBytes < LogShatterUtil.mbToBytes(readBatchSizeMb)) {
                line = logReader.readLine();
                batchSizeBytes = logReader.getDataPosition() - startPosition;

                if (line == null) {
                    filePosition = logReader.getFilePosition();
                    if (fileContext.getReaderFilePosition() != filePosition) {
                        fileContext.setReaderFilePosition(filePosition);
                    } else {
                        filePosition = -1;
                    }
                    break;
                }
                lines.add(line);
            }

            //В ситуации когда позиция ещё не извесна, но filePosition ещё не сохранен, посылаем пустой батч,
            //что бы записать позицию
            if (lines.isEmpty() && filePosition == -1) {
                return null;
            }
            long endPosition = logReader.getDataPosition();
            fileContext.setReaderDataPosition(endPosition);

            return new LogBatch(
                lines.stream(),
                endPosition,
                filePosition,
                batchSizeBytes,
                stopwatch.elapsed(),
                fileContext.getLogParser().getTableDescription().getColumns(),
                fileContext.getName()
            );
        }

    }

    private boolean openLogReader(FileContext fileContext) throws IOException {
        if (fileContext.getLogReader() != null) {
            return true;
        }
        Path file = fileContext.getPath();
        log.info("Opening logReader for file: " + file);
        readSemaphore.acquireOpenFileSemaphore();

        LogReader logReader;
        try {
            if (fileContext.getName().endsWith(".gz")) {
                logReader = new GzipFileLogReader(file);
            } else {
                logReader = new BufferedFileLogReader(file);
            }
        } catch (IOException e) {
            readSemaphore.releaseOpenFileSemaphore();
            throw e;
        }
        fileContext.setLogReader(logReader);
        BasicFileAttributes attrs = Files.readAttributes(file, BasicFileAttributes.class);
        openedFiles.put(fileContext, fileContext);
        if (!attrs.fileKey().equals(fileContext.getFileKey())) {
            log.warn(
                "Opened another file. (Log rotation?). Expected: " + fileContext.getFileKey() +
                    ", actual: " + attrs.fileKey() + ", fileName: " + file
            );
            fileContext.setRemoved(true);
            closeLogReader(fileContext);
            return false;
        }
        openedFiles.put(fileContext, fileContext);
        logReader.seek(fileContext.getReaderDataPosition());
        return true;
    }

    private void closeUselessFiles() {
        for (FileContext fileContext : openedFiles.keySet()) {
            maybeCloseLogReader(fileContext);
        }
    }

    private void maybeCloseLogReader(FileContext fileContext) {
        if (fileContext.getLogReader() == null) {
            return;
        }
        if (fileContext.isReading()) {
            return;
        }
        if (fileContext.isClosed()) {
            closeLogReader(fileContext);
            return;
        }
        if (readSemaphore.hasOpenFilesHunger()) {
            log.warn("Closing logReader because of queue to open new files");
            closeLogReader(fileContext);
            return;
        }

        if (fileContext.isFileUpdated()) {
            return;
        }
        long timeSinceModification = System.currentTimeMillis() - fileContext.getKnowLastModified();
        if (timeSinceModification > TimeUnit.MINUTES.toMillis(fileTimeSinceModificationToCloseMinutes)) {
            log.info(
                "Closing logReader for processed and unmodified for " + fileTimeSinceModificationToCloseMinutes +
                    " minutes file: " + fileContext.getName()
            );
            closeLogReader(fileContext);
        }
    }

    private void closeLogReader(FileContext fileContext) {
        LogReader logReader = fileContext.getLogReader();
        if (logReader == null) {
            return;
        }
        try {
            fileContext.setLogReader(null);
            logReader.close();
            log.info("LogReader closed for file: " + fileContext.getName());
        } catch (IOException e) {
            log.error("Problems with closing logReader", e);
        }
        openedFiles.remove(fileContext);
        readSemaphore.releaseOpenFileSemaphore();
    }

    @Required
    public void setFileWatcherService(FileWatcherService fileWatcherService) {
        this.fileWatcherService = fileWatcherService;
    }

    public void setReadBatchSizeMb(int readBatchSizeMb) {
        this.readBatchSizeMb = readBatchSizeMb;
    }

    public void setFileTimeSinceModificationToCloseMinutes(int fileTimeSinceModificationToCloseMinutes) {
        this.fileTimeSinceModificationToCloseMinutes = fileTimeSinceModificationToCloseMinutes;
    }

    public void setReadThreadCount(int readThreadCount) {
        this.readThreadCount = readThreadCount;
    }


}
