package ru.yandex.market.logshatter.reader.file;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.logshatter.config.ConfigurationService;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.config.LogSource;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.meta.LogshatterMetaDao;
import ru.yandex.market.logshatter.meta.SourceKey;
import ru.yandex.market.logshatter.meta.SourceMeta;
import ru.yandex.market.logshatter.reader.AbstractReaderService;

import java.io.IOException;
import java.net.InetAddress;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.*;
import java.util.concurrent.atomic.AtomicLong;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 19/02/15
 */
public class FileWatcherService extends AbstractReaderService implements InitializingBean {

    private static final Logger log = LogManager.getLogger();
    private static final String FILE_SCHEMA = "dir";

    private ConfigurationService configurationService;
    private LogshatterMetaDao logshatterMetaDao;
    private BatchErrorLoggerFactory errorLoggerFactory;

    private Map<Object, FileKeyContext> fileKeysContexts = new HashMap<>();

    private ListMultimap<Path, ConfigWrapper> logDirToConfigs = ArrayListMultimap.create();
    //Сразу загружаем информацию для известных файлов,
    //но используем её только один раз потом данные могут быть не актуальны
    public Map<SourceKey, SourceMeta> sourceMetas;
    private String origin;


    @Override
    public void afterPropertiesSet() throws Exception {
        origin = InetAddress.getLocalHost().getHostName();
        sourceMetas = logshatterMetaDao.getByOrigin(origin);
        Multimap<LogSource, LogShatterConfig> configs = configurationService.getConfigsForSchema(FILE_SCHEMA);
        for (Map.Entry<LogSource, Collection<LogShatterConfig>> sourceToConfigs : configs.asMap().entrySet()) {
            LogSource source = sourceToConfigs.getKey();
            Path logDir = Paths.get(source.getPath());
            for (LogShatterConfig config : sourceToConfigs.getValue()) {
                PathMatcher matcher = LogShatterUtil.createGlobMatcher(
                    logDir.toString() + "/" + config.getLogHosts() + "/" + config.getLogPath()
                );
                ConfigWrapper configWrapper = new ConfigWrapper(source, logDir, config, matcher);
                logDirToConfigs.put(logDir, configWrapper);
            }
        }

    }

    public synchronized List<FileContext> processAll() throws IOException {
        List<FileContext> fileContexts = new ArrayList<>();
        for (Path logDir : logDirToConfigs.keySet()) {
            fileContexts.addAll(processDir(logDir, logDirToConfigs.get(logDir)));
        }
        return fileContexts;
    }

    private List<FileContext> processDir(final Path dir, final List<ConfigWrapper> configs) throws IOException {

        long start = System.currentTimeMillis();

        final AtomicLong filesCount = new AtomicLong();

        final List<FileContext> updatedContexts = new ArrayList<>();

        Files.walkFileTree(dir, new FileVisitor<Path>() {
            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                filesCount.getAndIncrement();
                updatedContexts.addAll(processFile(file, attrs, configs));
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }
        });

        long timeMillis = System.currentTimeMillis() - start;
        log.info(
            "Processed dir: " + dir.toString() + ", files count:" + filesCount.get() + ", time " + timeMillis + "ms" +
                ", new/updated file contexts: " + updatedContexts.size()
        );
        return updatedContexts;
    }

    private List<FileContext> processFile(Path file, BasicFileAttributes attrs, List<ConfigWrapper> configs) {
        Object fileKey = attrs.fileKey();
        FileKeyContext fileKeyContext = fileKeysContexts.get(fileKey);
        if (fileKeyContext == null) {
            fileKeyContext = createFileKeyContext(fileKey, file, configs);
            fileKeysContexts.put(fileKeyContext.fileKey, fileKeyContext);
        }
        if (!fileKeyContext.file.equals(file)) {
            fileKeyContext = processRotation(fileKeyContext, file, configs);
            fileKeysContexts.put(fileKeyContext.fileKey, fileKeyContext);
        }

        if (fileKeyContext.useful) {
            return checkForUpdates(attrs, fileKeyContext.fileContexts);
        } else {
            return Collections.emptyList();
        }
    }

    private FileKeyContext processRotation(FileKeyContext fileKeyContext, Path newFile, List<ConfigWrapper> configs) {
        Path oldFile = fileKeyContext.file;

        if (isNewFile(oldFile, newFile)) {
            log.info(
                "Filename changed. Old: " + oldFile + ", new: " + newFile +
                    ", fileKey:" + fileKeyContext.fileKey + ", handling as new file."
            );
            for (FileContext fileContext : fileKeyContext.fileContexts) {
                fileContext.setRemoved(true);
                logshatterMetaDao.delete(fileContext);
            }
            return createFileKeyContext(fileKeyContext.fileKey, newFile, configs);
        } else {
            log.info(
                "Filename changed. Old: " + oldFile + ", new: " + newFile +
                    ", fileKey:" + fileKeyContext.fileKey + ", handling as rotation."
            );
            for (FileContext fileContext : fileKeyContext.fileContexts) {
                fileContext.setPath(newFile);
                logshatterMetaDao.save(fileContext);
            }
            return new FileKeyContext(fileKeyContext.fileKey, newFile, fileKeyContext.fileContexts);
        }
    }

    private List<FileContext> checkForUpdates(BasicFileAttributes attrs, List<FileContext> fileContexts) {
        List<FileContext> updatedContexts = new ArrayList<>();

        for (FileContext fileContext : fileContexts) {
            fileContext.setKnowLastModified(attrs.lastModifiedTime().toMillis());
            if (isModified(fileContext, attrs)) {
                fileContext.setKnownSizeBytes(attrs.size());
                updatedContexts.add(fileContext);
            }
        }
        return updatedContexts;
    }

    private boolean isNewFile(Path oldPath, Path newPath) {
        String oldName = oldPath.getName(oldPath.getNameCount() - 1).toString();
        String newName = newPath.getName(newPath.getNameCount() - 1).toString();
        if (oldName.equals(newName)) {
            return true; //Dir changed
        }
        int newNumber = getFileNumber(newName);
        int oldNumber = getFileNumber(oldName);
        return newNumber <= oldNumber;
    }

    public static int getFileNumber(String name) {
        if (name.endsWith(".gz")) {
            name = name.substring(0, name.length() - 3);
        }
        int dotPos = name.lastIndexOf('.');
        if (dotPos < 0) {
            return 0;
        }
        String numberString = name.substring(dotPos + 1, name.length());
        if (numberString.isEmpty()) {
            return 0;
        }
        try {
            return Integer.parseInt(numberString);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    private FileKeyContext createFileKeyContext(Object fileKey, Path file, List<ConfigWrapper> configs) {
        log.info("Found new file: " + file.toString() + ", fileKey: " + fileKey.toString());
        List<FileContext> fileContexts = new ArrayList<>();
        for (ConfigWrapper config : configs) {
            if (config.matcher.matches(file)) {
                log.info(
                    "Found configuration for file: " + file.toString() + ", fileKey: " + fileKey.toString() +
                        ", config: " + config.config.toString()
                );
                FileContext fileContext = createFileContext(fileKey, file, config);
                fileContexts.add(fileContext);
            }
        }
        //Берем имя из fileContext, что бы отработала ротация
        for (FileContext fileContext : fileContexts) {
            file = fileContext.getPath();
        }
        if (fileContexts.isEmpty()) {
            log.info("No configuration for file: " + file.toString() + ", fileKey: " + fileKey.toString());
        }
        return new FileKeyContext(fileKey, file, fileContexts);
    }

    private FileContext createFileContext(Object fileKey, Path file, ConfigWrapper config) {


        String hostname = file.getName(config.logDir.getNameCount()).toString();

        FileContext fileContext = new FileContext(
            config.config, origin, file, fileKey, hostname, errorLoggerFactory, readSemaphore.getEmptyQueuesCounter()
        );
        loadFileInfo(fileContext);
        return fileContext;

    }

    public void loadFileInfo(final FileContext fileContext) {
        SourceMeta sourceMeta = sourceMetas.remove(fileContext.getSourceKey());
        if (sourceMeta == null) {
            return;
        }
        String knowFileName = sourceMeta.getName();
        if (knowFileName != null && !fileContext.getName().equals(sourceMeta.getName())) {
            log.info("File " + fileContext.getName() + ", known as: " + knowFileName);
            //Ставим старое имя, потом отрабатываем ротацию
            fileContext.setPath(Paths.get(knowFileName));
        }
        fileContext.setDataOffset(sourceMeta.getDataOffset());
        fileContext.setReaderDataPosition(sourceMeta.getDataOffset());
        fileContext.setFileOffset(sourceMeta.getFileOffset());
    }

    private boolean isModified(FileContext fileContext, BasicFileAttributes attrs) {
        if (fileContext.getKnownSizeBytes() < attrs.size()) {
            return true;
        }
        return false;
    }

    @Required
    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    @Required
    public void setLogshatterMetaDao(LogshatterMetaDao logshatterMetaDao) {
        this.logshatterMetaDao = logshatterMetaDao;
    }

    public void setErrorLoggerFactory(BatchErrorLoggerFactory errorLoggerFactory) {
        this.errorLoggerFactory = errorLoggerFactory;
    }

    private static class FileKeyContext {
        private final Object fileKey;
        private final Path file;
        private final List<FileContext> fileContexts;
        private final boolean useful;

        public FileKeyContext(Object fileKey, Path file, List<FileContext> fileContexts) {
            this.fileKey = fileKey;
            this.file = file;
            this.fileContexts = fileContexts;
            this.useful = !fileContexts.isEmpty();
        }
    }

    private static class ConfigWrapper {
        private final LogSource source;
        private final Path logDir;
        private final LogShatterConfig config;
        private final PathMatcher matcher;

        public ConfigWrapper(LogSource source, Path logDir, LogShatterConfig config, PathMatcher matcher) {
            this.source = source;
            this.logDir = logDir;
            this.config = config;
            this.matcher = matcher;
        }
    }


}
