package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.base.Splitter;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.SetMultimap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.config.LogSource;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 14.09.2018
 */
public class LogBrokerConfigurationService {
    private static final Logger log = LogManager.getLogger();

    private static final String LOGBROKER_SCHEMA = "logbroker";

    private final MultiMap<LogbrokerSource, LogShatterConfig> sourcesToConfigs;
    private final List<LogbrokerSource> sources;
    private final SetMultimap<String, LogbrokerSource> identToSources;

    public LogBrokerConfigurationService(List<LogShatterConfig> configs, String disabledSourcesString) {
        this.sourcesToConfigs = createSourceToConfigsMap(configs, parseLogbrokerSources(disabledSourcesString));
        this.sources = createSourcesList(sourcesToConfigs);
        this.identToSources = createIdentToSourcesMap(sources);
        log.info("Found {} sources to read: {}", sources.size(), sources);
    }

    private static MultiMap<LogbrokerSource, LogShatterConfig> createSourceToConfigsMap(
        List<LogShatterConfig> configs, List<LogbrokerSource> disabledSources
    ) {
        MultiMap<LogbrokerSource, LogShatterConfig> result = new MultiMap<>();
        for (LogShatterConfig config : configs) {
            for (LogSource source : config.getSources()) {
                if (source.getSchema().equals(LOGBROKER_SCHEMA)) {
                    LogbrokerSource logbrokerSource = LogbrokerSource.fromSourcePath(source.getPath());

                    Optional<LogbrokerSource> disabledSource = disabledSources.stream()
                        .filter(s -> s.includes(logbrokerSource))
                        .findFirst();

                    if (disabledSource.isPresent()) {
                        log.warn(
                            "Skipping config {} cause source {} is disabled.",
                            config.getConfigFileName(), disabledSource.toString()
                        );
                        continue;
                    }
                    result.append(logbrokerSource, config);
                }
            }
        }
        return result;
    }

    private static List<LogbrokerSource> createSourcesList(MultiMap<LogbrokerSource, LogShatterConfig> sourcesToConfigs) {
        // Мы вычищаем те источники, которые покрываются более общими источниками.
        // Проще на примере: допустим, пользователи в разных конфигах написали разные source'ы:
        // * config1.json: market-health-stable--logtype1
        // * config2.json: market-health-stable--logtype2
        // * config3.json: market-health-stable
        // Источник в третьем конфиге покрывает источники из config1 и config2 и нам достаточно читать все партиции
        // из market-health-stable, поэтому мы вычищаем более частные случаи.
        List<LogbrokerSource> result = new ArrayList<>(sourcesToConfigs.keySet());
        result.removeIf(
            source ->
                source.getLogType() != null && sourcesToConfigs.containsKey(new LogbrokerSource(source.getIdent()))
        );
        return result;
    }

    private static HashMultimap<String, LogbrokerSource> createIdentToSourcesMap(List<LogbrokerSource> sources) {
        HashMultimap<String, LogbrokerSource> result = HashMultimap.create();
        for (LogbrokerSource source : sources) {
            result.put(source.getIdent(), source);
        }
        return result;
    }

    private static List<LogbrokerSource> parseLogbrokerSources(CharSequence string) {
        return Splitter.on(",")
            .trimResults()
            .omitEmptyStrings()
            .splitToList(string)
            .stream()
            .map(LogbrokerSource::fromSourcePath)
            .collect(Collectors.toList());
    }

    public List<LogbrokerSource> getSources() {
        return sources;
    }

    public MultiMap<LogbrokerSource, LogShatterConfig> getSourcesToConfigs() {
        return sourcesToConfigs;
    }

    public SetMultimap<String, LogbrokerSource> getIdentToSources() {
        return identToSources;
    }
}
