package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.collect.Ordering;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.logbroker.pull.LogBrokerSourceKey;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.logging.BatchErrorLoggerFactory;
import ru.yandex.market.logshatter.meta.LogshatterMetaDao;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.SourceContext;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.LogBrokerSourcesWithoutMetadataMonitoring;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author Alexander Kedrik <a href="mailto:alkedr@yandex-team.ru"></a>
 * @date 24.09.2018
 */
public class LogBrokerPartitionSourceContextsStorage {
    protected static final Logger log = LogManager.getLogger();

    private final MultiMap<LogBrokerSourceKey, LogbrokerSourceContext> sourceKeyToContext = new MultiMap<>();

    private final MultiMap<LogbrokerSource, LogShatterConfig> sourcesToConfigs;
    private final ReadSemaphore readSemaphore;
    private final LogshatterMetaDao logshatterMetaDao;
    private final LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring;
    private final BatchErrorLoggerFactory errorLoggerFactory;

    public LogBrokerPartitionSourceContextsStorage(
        LogBrokerConfigurationService logBrokerConfigurationService,
        ReadSemaphore readSemaphore,
        LogshatterMetaDao logshatterMetaDao,
        LogBrokerSourcesWithoutMetadataMonitoring sourcesWithoutMetadataMonitoring,
        BatchErrorLoggerFactory errorLoggerFactory
    ) {
        this.sourcesToConfigs = logBrokerConfigurationService.getSourcesToConfigs();
        this.readSemaphore = readSemaphore;
        this.logshatterMetaDao = logshatterMetaDao;
        this.sourcesWithoutMetadataMonitoring = sourcesWithoutMetadataMonitoring;
        this.errorLoggerFactory = errorLoggerFactory;
    }

    public synchronized void reloadAll() {
        for (List<LogbrokerSourceContext> sourceContexts : sourceKeyToContext.values()) {
            for (LogbrokerSourceContext sourceContext : sourceContexts) {
                logshatterMetaDao.load(sourceContext);
                sourceContext.resetFinishedState();
            }
        }
    }

    public synchronized CompletableFuture<Void> finishAll() {
        return CompletableFuture.allOf(
            sourceKeyToContext.values().stream()
                .flatMap(Collection::stream)
                .map(LogbrokerSourceContext::finish)
                .toArray(CompletableFuture<?>[]::new)
        );
    }

    public synchronized List<ImmutablePair<LogbrokerSourceContext, Long>> getSourceContextsWithOldestLogBatches(
        Function<SourceContext, Long> sourceContextToCreationTimeOfTheLastBatchFunction
    ) {
        return Ordering.from(Comparator.<ImmutablePair<LogbrokerSourceContext, Long>, Long>comparing(ImmutablePair::getRight))
            .leastOf(
                sourceKeyToContext.values().stream()
                    .flatMap(Collection::stream)
                    .map(sourceContext -> new ImmutablePair<>(
                        sourceContext,
                        sourceContextToCreationTimeOfTheLastBatchFunction.apply(sourceContext)
                    ))
                    .filter(pair -> pair.getRight() < Long.MAX_VALUE)
                    .collect(Collectors.toList()),
                5
            );
    }

    public synchronized List<LogbrokerSourceContext> getSourceContexts(LogBrokerSourceKey key, int instanceId) {
        List<LogbrokerSourceContext> fileContexts = sourceKeyToContext.get(key);
        if (fileContexts == null) {
            fileContexts = createSourceContexts(key, instanceId);

            if (fileContexts.isEmpty() && !key.hashSourceInfo()) {
                sourcesWithoutMetadataMonitoring.notifyAboutSourceWithoutMetadata(key);
                return Collections.emptyList();
            }

            sourceKeyToContext.put(key, fileContexts);
        }
        return fileContexts;
    }

    private List<LogbrokerSourceContext> createSourceContexts(LogBrokerSourceKey logBrokerSourceKey, int instanceId) {
        List<LogShatterConfig> configsByIdentOnly = sourcesToConfigs.get(
            new LogbrokerSource(logBrokerSourceKey.getIdent())
        );

        List<LogShatterConfig> configsByTopic = sourcesToConfigs.get(
            LogbrokerSource.fromTopic(logBrokerSourceKey.getTopic())
        );

        if (configsByIdentOnly == null && configsByTopic == null) {
            log.warn("Unknown ident: {} for: {}", logBrokerSourceKey.getIdent(), logBrokerSourceKey);
            return Collections.emptyList();
        }

        List<LogShatterConfig> configs = new ArrayList<>();
        if (configsByIdentOnly != null) {
            configs.addAll(configsByIdentOnly);
        }
        if (configsByTopic != null) {
            configs.addAll(configsByTopic);
        }

        List<LogbrokerSourceContext> fileContexts = new ArrayList<>();
        for (LogShatterConfig config : configs) {
            boolean noMetaAndAnySource = !logBrokerSourceKey.hashSourceInfo() && config.isLogHostsWild() &&
                config.isLogPathWild();
            boolean pathOrHostMatches = logBrokerSourceKey.hashSourceInfo() &&
                config.getHostMatcher().matches(Paths.get(logBrokerSourceKey.getServer())) &&
                config.getFileMatcher().matches(Paths.get(logBrokerSourceKey.getPath()));
            if (noMetaAndAnySource || pathOrHostMatches) {
                Path path = noMetaAndAnySource ? null : Paths.get(logBrokerSourceKey.getPath());
                fileContexts.add(createSourceContext(path, config, logBrokerSourceKey, instanceId));
            }
        }

        if (fileContexts.isEmpty()) {
            log.info("No configs found for: {}", logBrokerSourceKey);
            return Collections.emptyList();
        } else {
            log.info(
                "Found {} configs for: {}. Configs: {}",
                fileContexts.size(), logBrokerSourceKey,
                fileContexts.stream()
                    .map(c -> c.getLogShatterConfig().getConfigFileName())
                    .collect(Collectors.joining(", "))
            );
        }

        return fileContexts;
    }

    private LogbrokerSourceContext createSourceContext(Path path, LogShatterConfig config,
                                                       LogBrokerSourceKey logBrokerSourceKey, int instanceId) {
        LogbrokerSourceContext sourceContext = new LogbrokerSourceContext(
            path,
            config,
            logBrokerSourceKey,
            errorLoggerFactory,
            instanceId,
            readSemaphore.getQueuesCounterForSource(
                String.format("%s--%s", logBrokerSourceKey.getIdent(), logBrokerSourceKey.getType())
            )
        );

        logshatterMetaDao.load(sourceContext);

        return sourceContext;
    }
}
