package ru.yandex.market.logshatter.reader.logbroker;

import com.google.common.annotations.VisibleForTesting;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.logbroker.pull.LogBrokerClient;
import ru.yandex.market.logbroker.pull.LogBrokerOffset;
import ru.yandex.market.logbroker.pull.LogBrokerSession;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.reader.AbstractReaderService;
import ru.yandex.market.logshatter.reader.logbroker.monitoring.MonitoringConfig;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 01/07/15
 */
public class LogBrokerReaderService extends AbstractReaderService implements InitializingBean, Runnable {
    private static final int BYTES_IN_MB = 1024 * 1024;
    private static final double FACTOR_TO_CONSIDER_PARTITION_HAS_DATA = 0.8;

    private static final Logger log = LogManager.getLogger();

    private static final String LOGBROKER_SCHEMA = "logbroker";

    private final LogBrokerClient logBrokerClient;
    private final PartitionDao partitionDao;
    private final PartitionManager partitionManager;
    private final MonitoringConfig monitoringConfig;

    private MonitoringUnit logBrokerReaderServiceUnit;

    private final List<LogbrokerSource> sources;

    private int sleepAfterErrorSeconds = 5;
    private int processIntervalSeconds = 60;
    private int logbrokerReaderThreads = 5;
    private int maxMessagesPerRead = 10_000;
    private int maxBytesPerRead = 50 * BYTES_IN_MB;


    private final MultiMap<LogbrokerSource, LogShatterConfig> sourcesToConfigs;

    public LogBrokerReaderService(LogBrokerClient logBrokerClient, PartitionDao partitionDao,
                                  PartitionManager partitionManager, MonitoringConfig monitoringConfig,
                                  LogBrokerConfigurationService logBrokerConfigurationService) {
        this.logBrokerClient = logBrokerClient;
        this.partitionDao = partitionDao;
        this.partitionManager = partitionManager;
        this.monitoringConfig = monitoringConfig;
        this.monitoring = monitoringConfig.getMonitoring();

        this.sources = logBrokerConfigurationService.getSources();
        this.sourcesToConfigs = logBrokerConfigurationService.getSourcesToConfigs();
    }

    @Override
    public void afterPropertiesSet() {
        logBrokerReaderServiceUnit = new MonitoringUnit("LogBrokerReaderService — " + LOGBROKER_SCHEMA);
        logBrokerReaderServiceUnit.setMonitoringDelay(monitoringConfig.getReaderMonitoringDelayMinutes(), TimeUnit.MINUTES);

        monitoring.getClusterCritical().addUnit(logBrokerReaderServiceUnit);

        if (sources.isEmpty()) {
            log.info("No logbroker sources. LogBrokerReaderService will not start.");
            return;
        }

        ExecutorService logbrokerExecutorService = createExecutorService(logbrokerReaderThreads, LOGBROKER_SCHEMA + "-");
        for (int i = 0; i < logbrokerReaderThreads; i++) {
            logbrokerExecutorService.submit(new ReaderWorker());
        }

        new Thread(this, LOGBROKER_SCHEMA + " master").start();
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            boolean isError = false;
            try {
                partitionManager.releaseExtraLocks();
            } catch (Exception e) {
                String message = "Exception in Logbroker monitoring";
                log.error(message, e);
                logBrokerReaderServiceUnit.critical(message, e);
                isError = true;
            }

            if (!isError) {
                logBrokerReaderServiceUnit.ok();
            }

            try {
                TimeUnit.SECONDS.sleep(processIntervalSeconds);
            } catch (InterruptedException ignored) {
            }
        }
    }

    @VisibleForTesting
    protected List<LogbrokerSource> getSources() {
        return sources;
    }

    private class ReaderWorker implements Runnable {
        @Override
        public void run() {
            while (!Thread.interrupted()) {
                try {
                    read();
                } catch (InterruptedException ignored) {
                }
            }
        }

        private void read() throws InterruptedException {
            boolean error = false;
            PartitionContext partition;
            try {
                partition = partitionManager.acquirePartition();
            } catch (Exception e) {
                log.error("Failed to select partition", e);
                TimeUnit.SECONDS.sleep(sleepAfterErrorSeconds);
                return;
            }
            try {
                read(partition);
            } catch (Exception e) {
                log.error("Error while reading from logbroker. Session: " + partition.getSession(), e);
                error = true;
            } finally {
                partitionManager.releasePartition(partition, error);
                if (error) {
                    TimeUnit.SECONDS.sleep(sleepAfterErrorSeconds);
                }
            }

        }

        private LogBrokerOffset createOffset(PartitionContext partition, long offset, String dc) {
            return new LogBrokerOffset(
                partition.getName(), offset, partition.getLogStart(),
                partition.getLogEnd(), partition.getLag(), partition.getOwner(), dc
            );
        }

        private void read(final PartitionContext partition) throws InterruptedException, IOException {
            if (!partition.hasSession()) {
                LogBrokerSession session = logBrokerClient.openSession(partition.getName(), partition.getHost());
                partition.setSession(session);
                partition.clearOffsetsState();
            }

            LogBrokerOffset offset = partitionDao.get(partition.getName());
            partition.setCommittableOffset(offset.getOffset());
            commit(partition);

            partition.maybeReloadSourceContexts();
            while (!Thread.interrupted() && partition.isReadAllowed()) {
                readSemaphore.waitForRead();
                boolean dataRemainingInPartition = doRead(partition);
                // коммитим чаще, чтобы логировать актуальный лаг, а не ждать чтения всей партиции.
                commit(partition);
                if (!dataRemainingInPartition) {
                    break; //Достигли конца, пора читать другие
                }
            }
        }

        /**
         * @return true if partition still has data, false if reach the end
         */
        private boolean doRead(final PartitionContext partition) throws IOException {
            final AtomicInteger messageCount = new AtomicInteger();
            final AtomicInteger bytesCount = new AtomicInteger();

            logBrokerClient.read(
                partition.getSession(),
                (session, meta, inputStream, receiveTimeMillis) -> {
                    messageCount.incrementAndGet();
                    bytesCount.addAndGet(meta.getSize());
                    partition.addDataToParseQueues(meta, inputStream);
                },
                maxMessagesPerRead, maxBytesPerRead
            );
            return isDataRemainingInPartition(messageCount.get(), bytesCount.get());
        }

        private boolean isDataRemainingInPartition(int readMessageCount, int readBytesCount) {
            if (readMessageCount >= maxMessagesPerRead) {
                return true;
            }

            if (readBytesCount >= maxBytesPerRead * FACTOR_TO_CONSIDER_PARTITION_HAS_DATA) {
                //Можем прочитать и логброкера меньше лимита,
                //т.к. сообщения добираются до maxBytesPerRead и обычно меньше
                //Поэтому считаем, что если мы прочитали 80% от лимита - там ещё есть данные.
                return true;
            }

            return false;
        }

        private void softCommit(PartitionContext partition) throws IOException {
            if (partition.getSoftOffset() <= 0) {
                return;
            }
            logBrokerClient.commit(partition.getSession(), partition.getSoftOffset(), true);
        }

        private void commit(PartitionContext partition) throws IOException {
            long offset = partition.getCommittableOffset();
            if (offset > 0 && partition.getLastCommitedOffset() < offset) {
                partitionDao.save(createOffset(partition, offset, logBrokerClient.getDc()));
                logBrokerClient.commit(partition.getSession(), offset, false);
                log.debug("Committed offset for " + partition.getName() + " ::: " + offset);
                partition.setLastCommitedOffset(offset);
            }
            if (offset < partition.getSoftOffset()) {
                softCommit(partition); //Т.к. hard перетирает позицию soft.
            }
        }
    }

    public void setSleepAfterErrorSeconds(int sleepAfterErrorSeconds) {
        this.sleepAfterErrorSeconds = sleepAfterErrorSeconds;
    }

    public void setProcessIntervalSeconds(int processIntervalSeconds) {
        this.processIntervalSeconds = processIntervalSeconds;
    }

    public void setLogbrokerReaderThreads(int logbrokerReaderThreads) {
        this.logbrokerReaderThreads = logbrokerReaderThreads;
    }


    public void setMaxMessagesPerRead(int maxMessagesPerRead) {
        this.maxMessagesPerRead = maxMessagesPerRead;
    }

    public void setMaxMbPerRead(int maxMbPerRead) {
        this.maxBytesPerRead = maxMbPerRead * BYTES_IN_MB;
    }
}
